/* -*- C++ -*-
 * Copyright (c) 1995-1996 Tohru Kisuki
 *               1995-2001 Masaki Wakabayashi
 *               1998 Keisuke Inoue
 *      	 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */
/*
 * <<< bus_port_shared.h >>>
 * bus port for shared bus
 */

#ifndef BUS_PORT_SHARED_H
#define BUS_PORT_SHARED_H

#ifdef DEBUG
# include <cassert>
#endif
#include <isis/isis.h>
#include "bus_packet_shared.h"

template <class Type>
class bus_port_shared : public bus_port_base<Type>
{
private:
	typedef bus_port_base<Type> inherited;
	typedef bus_packet_base<Type> abst_packet_t;
	typedef bus_packet_shared<Type> packet_t;
	const packet_t* inherited_look() const {
		return (packet_t*)(inherited::look()); }
	packet_t* inherited_look() {
		return (packet_t*)(inherited::look()); }
#ifndef DEBUG
	const packet_t& inherited_packet() const {
		return *inherited_look(); }
	packet_t& inherited_packet() {
		return *inherited_look(); }
#else // DEBUG
	const packet_t& inherited_packet() const {
		assert(inherited_look() != NULL);  return *inherited_look(); }
	packet_t& inherited_packet() {
		assert(inherited_look() != NULL);  return *inherited_look(); }
#endif // DEBUG
public:
	bus_port_shared();
	bus_port_shared(const bus_port_shared<Type>&);
	virtual ~bus_port_shared();
	virtual void output_to_ostream(ostream&) const;
	// send_single_{read,write}_{request,grant,ack,nack,data}()
	inline void send_single_read_request(Type);
	inline void send_single_read_grant();
	inline void send_single_read_ack();
	inline void send_single_read_nack();
	inline void send_single_read_data(Type);
	inline void send_single_write_request(Type);
	inline void send_single_write_grant();
	inline void send_single_write_ack();
	inline void send_single_write_nack();
	inline void send_single_write_data(Type);
	// send_multi_{read,write}_{request,grant,ack,nack,data}()
	inline void send_multi_read_request(Type, unsigned);
	inline void send_multi_read_grant();
	inline void send_multi_read_ack();
	inline void send_multi_read_nack();
	inline void send_multi_read_data(Type, unsigned, unsigned);
	inline void send_multi_write_request(Type, unsigned);
	inline void send_multi_write_grant();
	inline void send_multi_write_ack();
	inline void send_multi_write_nack();
	inline void send_multi_write_data(Type, unsigned, unsigned);
	// for snoop cache protocol
	int is_update_request() const {
		return have_packet() && inherited_packet().is_update_request();}
	int is_update_ack() const {
		return have_packet() && inherited_packet().is_update_ack();}
	int is_invalidation_request() const {
		return have_packet() && inherited_packet().is_invalidation_request(); }
	int is_line_copy_request() const {
		return have_packet() && inherited_packet().is_line_copy_request(); }
	int is_line_copy_ack() const {
		return have_packet() && inherited_packet().is_line_copy_ack(); }
	int is_line_move_request() const {
		return have_packet() && inherited_packet().is_line_move_request(); }
	int is_line_move_ack() const {
		return have_packet() && inherited_packet().is_line_move_ack(); }
//	inline void send_update_request();
//	inline void send_update_ack();
//	inline void send_update_ack(Type);
	inline void send_invalidation_request();
	inline void send_invalidation_request(Type);
	inline void send_line_copy_request(Type);
	inline void send_line_copy_ack();
	inline void send_line_move_request(Type);
	inline void send_line_move_ack();
};

template <class Type>
inline void bus_port_shared<Type>::send_single_read_request(Type a)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_single_read_request(a);
		put(p);
	} else {
		packet_reference().set_single_read_request(a);
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_single_read_ack()
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_single_read_ack();
		put(p);
	} else {
		packet_reference().set_single_read_ack();
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_single_read_nack()
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_single_read_nack();
		put(p);
	} else {
		packet_reference().set_single_read_nack();
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_single_read_data(Type a)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_single_read_data(a);
		put(p);
	} else {
		packet_reference().set_single_read_data(a);
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_single_write_request(Type a)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_single_write_request(a);
		put(p);
	} else {
		packet_reference().set_single_write_request(a);
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_single_write_ack()
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_single_write_ack();
		put(p);
	} else {
		packet_reference().set_single_write_ack();
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_single_write_nack()
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_single_write_nack();
		put(p);
	} else {
		packet_reference().set_single_write_nack();
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_single_write_data(Type a)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_single_write_data(a);
		put(p);
	} else {
		packet_reference().set_single_write_data(a);
	}
}

template <class Type>
inline void bus_port_shared<Type>::
send_multi_read_request(Type a, unsigned b)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_multi_read_request(a, b);
		put(p);
	} else {
		packet_reference().set_multi_read_request(a, b);
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_multi_read_ack()
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_multi_read_ack();
		put(p);
	} else {
		packet_reference().set_multi_read_ack();
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_multi_read_nack()
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_multi_read_nack();
		put(p);
	} else {
		packet_reference().set_multi_read_nack();
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_multi_read_data(Type a,
	unsigned b, unsigned c)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_multi_read_data(a, b, c);
		put(p);
	} else {
		packet_reference().set_multi_read_data(a, b, c);
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_multi_write_request(Type a,
	unsigned b)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_multi_write_request(a, b);
		put(p);
	} else {
		packet_reference().set_multi_write_request(a, b);
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_multi_write_ack()
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_multi_write_ack();
		put(p);
	} else {
		packet_reference().set_multi_write_ack();
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_multi_write_nack()
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_multi_write_nack();
		put(p);
	} else {
		packet_reference().set_multi_write_nack();
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_multi_write_data(Type a,
	unsigned b, unsigned c)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_multi_write_data(a, b, c);
		put(p);
	} else {
		packet_reference().set_multi_write_data(a, b, c);
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_invalidation_request(Type a)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_invalidation_request(a);
		put(p);
	} else {
		inherited_packet().set_invalidation_request(a);
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_line_copy_request(Type a)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_line_copy_request(a);
		put(p);
	} else {
		inherited_packet().set_line_copy_request(a);
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_line_copy_ack()
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_line_copy_ack();
		put(p);
	} else {
		inherited_packet().set_line_copy_ack();
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_line_move_request(Type a)
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_line_move_request(a);
		put(p);
	} else {
		inherited_packet().set_line_move_request(a);
	}
}

template <class Type>
inline void bus_port_shared<Type>::send_line_move_ack()
{
	if (!have_packet()) {
		packet_t* p = new packet_t;
		p->set_line_move_ack();
		put(p);
	} else {
		inherited_packet().set_line_move_ack();
	}
}

template <class Type>
bus_port_shared<Type>::bus_port_shared()
{}

template <class Type>
bus_port_shared<Type>::bus_port_shared(const bus_port_shared<Type>& a)
	: inherited(a)
{}

template <class Type>
bus_port_shared<Type>::~bus_port_shared()
{}

template <class Type>
void bus_port_shared<Type>::output_to_ostream(ostream& os) const
{
	if (!have_packet()) {
		os << "ready";
	} else {
		os << packet_reference();
	}
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif // BUS_PORT_SHARED_H
