/* -*- C++ -*-
 *
 * <<< shared_bus_interface_unit.h >>>
 *
 * --- Shared bus interface unit class 'shared_bus_interface_unit'
 *     Copyright (C) 1998-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef SHARED_BUS_INTERFACE_UNIT_H
#define SHARED_BUS_INTERFACE_UNIT_H 1

#include <iostream>
#include <isis/isis.h>

template <class A, class D = A>
class shared_bus_interface_unit : public synchronous_unit
{
private:
	typedef shared_bus_interface_unit thisclass;
	typedef synchronous_unit inherited;
public:
	typedef A address_type;
	typedef A size_type;
	typedef D data_type;
private:
	enum state_type {
		READY,
		COMMON_REQ_OWNER,
		COMMON_SEND,
		COMMON_WAIT,
		COMMON_ACK,
		COMMON_NACK,
		SINGLE_READ_TRANS_WAIT,
		SINGLE_READ_TRANS,
		MULTI_READ_TRANS_WAIT,
		MULTI_READ_TRANS,
		SINGLE_WRITE_TRANS_WAIT,
		SINGLE_WRITE_TRANS,
		SINGLE_WRITE_END,
		MULTI_WRITE_TRANS_WAIT,
		MULTI_WRITE_TRANS,
		MULTI_WRITE_END
	};
	enum access_type {
		SINGLE_READ, SINGLE_WRITE, MULTI_READ, MULTI_WRITE
	};
	typedef bus_port<address_type, data_type> bus_port_type;
	bus_port_type lobus_if, shbus_if;
	state_type state;
	access_type access;
	size_t total_packet_count, packet_number, packet_count;
	address_type top_, adr_;
	size_type size_;
	data_type dat_;
public:
	shared_bus_interface_unit(void);
	shared_bus_interface_unit(const thisclass&);
	virtual ~shared_bus_interface_unit();
	virtual void output(ostream&) const;
	inline virtual void clock_in(void);
	inline virtual void clock_out(void);
	virtual void reset(void);
	const port& local_bus_port(void) const { return lobus_if; }
	const port& shared_bus_port(void) const { return shbus_if; }
	port& local_bus_port(void) { return lobus_if; }
	port& shared_bus_port(void) { return shbus_if; }
	virtual bool is_valid_address(address_type) const;
	void set_valid_address(address_type, address_type);
	void connect_local_bus_port(bus_port_type& a) { lobus_if.connect(a); }
	void connect_local_bus_port(single_port_synchronous_unit& a)
		{ a.port_ref().connect(lobus_if); }
	void disconnect_local_bus_port(void) { lobus_if.disconnect(); }
	void disconnect_local_bus_port(bus_port_type& a)
		{ lobus_if.disconnect(a); }
	void connect_shared_bus_port(bus_port_type& a) { shbus_if.connect(a); }
	void connect_shared_bus_port(single_port_synchronous_unit& a)
		{ a.port_ref().connect(shbus_if); }
	void disconnect_shared_bus_port(void) { shbus_if.disconnect(); }
	void disconnect_shared_bus_port(bus_port_type& a)
		{ shbus_if.disconnect(a); }
	bool is_connected_to_local_bus_port(void) const
		{ return lobus_if.is_connected(); }
	bool is_connected_to_local_bus_port(const bus_port_type& a) const
		{ return lobus_if.is_connected(a); }
	bool is_connected_to_shared_bus_port(void) const
		{ return shbus_if.is_connected(); }
	bool is_connected_to_shared_bus_port(const bus_port_type& a) const
		{ return shbus_if.is_connected(a); }
};

template <class A, class D>
shared_bus_interface_unit<A, D>::shared_bus_interface_unit(void)
	: state(READY),
	  top_(address_type(0)),
	  size_(size_type(0))
{}

template <class A, class D>
shared_bus_interface_unit<A, D>::shared_bus_interface_unit
	(const shared_bus_interface_unit& a)
	: state(READY),
	  top_(a.top_),
	  size_(a.size_)
{}

template <class A, class D>
shared_bus_interface_unit<A, D>::~shared_bus_interface_unit()
{}

template <class A, class D>
bool shared_bus_interface_unit<A, D>::is_valid_address
	(shared_bus_interface_unit<A, D>::address_type a) const
{
	return a >= top_ && a < top_ + size_;
}

template <class A, class D>
void shared_bus_interface_unit<A, D>::set_valid_address
	(shared_bus_interface_unit<A, D>::address_type a,
	 shared_bus_interface_unit<A, D>::size_type b)
{
	if (b > 0) top_ = a, size_ = b;
}

template <class A, class D>
void shared_bus_interface_unit<A, D>::reset(void)
{
	state = READY;
}

template <class A, class D>
void shared_bus_interface_unit<A, D>::clock_in(void)
{
	switch (state) {
	case READY:
		{
			if (lobus_if.is_ready() || shbus_if.is_owned()) break;
			address_type tmp = lobus_if.address();
			if (!is_valid_address(tmp)) break;
			if (lobus_if.is_single_read_request()) {
				adr_ = tmp;
				access = SINGLE_READ;
				state = COMMON_REQ_OWNER;
			} else if (lobus_if.is_multi_read_request()) {
				adr_ = tmp;
				total_packet_count = lobus_if.total_packet_count();
				packet_count = 0;
				access = MULTI_READ;
				state = COMMON_REQ_OWNER;
			} else if (lobus_if.is_single_write_request()) {
				adr_ = tmp;
				access = SINGLE_WRITE;
				state = COMMON_REQ_OWNER;
			} else if (lobus_if.is_multi_write_request()) {
				adr_ = tmp;
				total_packet_count = lobus_if.total_packet_count();
				packet_count = 0;
				access = MULTI_WRITE;
				state = COMMON_REQ_OWNER;
			}
		}
		break;
	case COMMON_REQ_OWNER:
		if (shbus_if.is_owner()) state = COMMON_SEND;
		break;
	case COMMON_WAIT:
		if (shbus_if.is_ack()) {
			state = COMMON_ACK;
		} else if (shbus_if.is_nack()) {
			state = COMMON_NACK;
		}
		break;
	case SINGLE_READ_TRANS_WAIT:
		if (shbus_if.is_single_read_data()) {
			dat_ = shbus_if.data();
			state = SINGLE_READ_TRANS;
		}
		break;
	case MULTI_READ_TRANS_WAIT:
		if (shbus_if.is_multi_read_data()) {
			dat_ = shbus_if.data();
			packet_number = shbus_if.packet_number();
			state = MULTI_READ_TRANS;
		}
		break;
	case MULTI_READ_TRANS:
		dat_ = shbus_if.data();
		packet_number = shbus_if.packet_number();
		break;
	case SINGLE_WRITE_TRANS_WAIT:
		if (lobus_if.is_single_write_data()) {
			dat_ = lobus_if.data();
			state = SINGLE_WRITE_TRANS;
		}
		break;
	case MULTI_WRITE_TRANS_WAIT:
		if (lobus_if.is_multi_write_data()) {
			dat_ = lobus_if.data();
			packet_number = lobus_if.packet_number();
			state = MULTI_WRITE_TRANS;
		}
		break;
	case MULTI_WRITE_TRANS:
		dat_ = lobus_if.data();
		packet_number = lobus_if.packet_number();
		break;
	default:
		break;
	}
}

template <class A, class D>
void shared_bus_interface_unit<A, D>::clock_out(void)
{
	switch (state) {
	case COMMON_REQ_OWNER:
		shbus_if.request_ownership();
		break;
	case COMMON_SEND:
		switch (access) {
		case SINGLE_READ:
			shbus_if.send_single_read_request(adr_);
			break;
		case MULTI_READ:
			shbus_if.send_multi_read_request(adr_, total_packet_count);
			break;
		case SINGLE_WRITE:
			shbus_if.send_single_write_request(adr_);
			break;
		case MULTI_WRITE:
			shbus_if.send_multi_write_request(adr_, total_packet_count);
			break;
		}
		state = COMMON_WAIT;
		break;
	case COMMON_ACK:
		switch (access) {
		case SINGLE_READ:
			lobus_if.send_single_read_ack();
			state = SINGLE_READ_TRANS_WAIT;
			break;
		case MULTI_READ:
			lobus_if.send_multi_read_ack();
			state = MULTI_READ_TRANS_WAIT;
			break;
		case SINGLE_WRITE:
			lobus_if.send_single_write_ack();
			state = SINGLE_WRITE_TRANS_WAIT;
			break;
		case MULTI_WRITE:
			lobus_if.send_multi_write_ack();
			state = MULTI_WRITE_TRANS_WAIT;
			break;
		}
		break;
	case COMMON_NACK:
		shbus_if.clear();
		shbus_if.release_ownership();
		state = COMMON_REQ_OWNER;
		break;
	case SINGLE_READ_TRANS:
		lobus_if.send_single_read_data(dat_);
		shbus_if.clear();
		shbus_if.release_ownership();
		state = READY;
		break;
	case MULTI_READ_TRANS:
		lobus_if.send_multi_read_data(dat_, total_packet_count, packet_number);
		if (++packet_count == total_packet_count) {
			shbus_if.clear();
			shbus_if.release_ownership();
			state = READY;
		}
		break;
	case SINGLE_WRITE_TRANS:
		shbus_if.send_single_write_data(dat_);
		state = SINGLE_WRITE_END;
		break;
	case SINGLE_WRITE_END:
		shbus_if.clear();
		shbus_if.release_ownership();
		state = READY;
		break;
	case MULTI_WRITE_TRANS:
		shbus_if.send_multi_write_data(dat_, total_packet_count, packet_number);
		if (++packet_count == total_packet_count) {
			lobus_if.clear();
			shbus_if.release_ownership();
			state = MULTI_WRITE_END;
		}
		break;
	case MULTI_WRITE_END:
		shbus_if.clear();
		shbus_if.release_ownership();
		state = READY;
		break;
	default:
		break;
	}
}

template <class A, class D>
void shared_bus_interface_unit<A, D>::output(ostream& os) const
{
	const long flags = os.flags();
	os << hex;
	switch (state) {
	case READY:
		os << "ready";
		break;
	case COMMON_REQ_OWNER:
		os << "common_request_ownership(0x" << adr_ << ')';
		break;
	case COMMON_SEND:
		os << "common_send_request(0x" << adr_ << ')';
		break;
	case COMMON_WAIT:
		os << "common_wait(0x" << adr_ << ')';
		break;
	case COMMON_ACK:
		os << "common_ack(0x" << adr_ << ')';
		break;
	case COMMON_NACK:
		os << "common_nack(0x" << adr_ << ')';
		break;
	case SINGLE_READ_TRANS_WAIT:
		os << "single_read_trans_wait(0x" << adr_ << ')';
		break;
	case SINGLE_READ_TRANS:
		os << "single_read_trans(0x" << adr_ << ')';
		break;
	case MULTI_READ_TRANS_WAIT:
		os << "multi_read_trans_wait(0x" << adr_ << ')';
		break;
	case MULTI_READ_TRANS:
		os << "multi_read_trans(0x" << adr_ << ')';
		break;
	case SINGLE_WRITE_TRANS_WAIT:
		os << "single_write_trans_wait(0x" << adr_ << ')';
		break;
	case SINGLE_WRITE_TRANS:
		os << "single_write_trans(0x" << adr_ << ')';
		break;
	case SINGLE_WRITE_END:
		os << "single_write_end(0x" << adr_ << ')';
		break;
	case MULTI_WRITE_TRANS_WAIT:
		os << "multi_write_trans_wait(0x" << adr_ << ')';
		break;
	case MULTI_WRITE_TRANS:
		os << "multi_write_trans(0x" << adr_ << ')';
		break;
	case MULTI_WRITE_END:
		os << "multi_write_end(0x" << adr_ << ')';
		break;
	}
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* SHARED_BUS_INTERFACE_UNIT_H */
