/* -*- C++ -*-
 *
 * <<< shared_bus_architecture.h >>>
 *
 * --- Shared bus architecture class 'shared_bus_architecture'
 *     Copyright (C) 1998-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef SHARED_BUS_ARCHITECTURE_H
#define SHARED_BUS_ARCHITECTURE_H 1

// #include <vector>
#include <isis/array.h>
#include <isis/isis.h>
#include "shared_bus_element.h"

class ostream;

class shared_bus_architecture : public multiprocessor<shared_bus_element>
{
private:
	typedef shared_bus_architecture thisclass;
	typedef multiprocessor<shared_bus_element> inherited;
public:
	typedef inherited::address_type address_type;
	typedef inherited::data_type data_type;
	typedef inherited::char_type char_type;
	typedef inherited::size_type size_type;
	typedef inherited::processor_type processor_type;
	typedef inherited::memory_map_type memory_map_type;
	typedef shared_bus_element processing_element_type;
	static const size_t sizeof_word_type = sizeof_r3000_word;
	typedef mapped_memory<address_type, data_type, sizeof_word_type, char_type>
		mapped_memory_type;
private:
	array<processing_element_type> pe;
	mapped_memory_type shmem, shfad;
	memory_control_unit<address_type, data_type, sizeof_word_type, char_type>
		shmem_ctl;
	fetch_and_dec_control_unit<address_type, data_type, sizeof_word_type,
							   char_type> shfad_ctl;
public:
	shared_bus_architecture(void);
	explicit shared_bus_architecture(size_t);
	shared_bus_architecture(const thisclass&);
	virtual ~shared_bus_architecture();
	virtual void output(ostream&) const;
	virtual const processing_element_type& processing_element(size_t i) const
		{ return pe[i]; }
	virtual processing_element_type& processing_element(size_t i)
		{ return pe[i]; }
	virtual size_t processor_number(void) const { return pe.size(); }
	virtual void set_processor_number(size_t);
	virtual bool is_halt(void) const;
	virtual bool is_bus_error(void) const;
	address_type shared_memory_address(void) const { return shmem.top(); }
	size_type shared_memory_size(void) const { return shmem.size(); }
	address_type sync_memory_address(void) const { return shfad.top(); }
	size_type sync_memory_size(void) const { return shfad.size(); }
	void set_shared_memory_area(address_type, size_type);
	void set_sync_memory_area(address_type, size_type);
	void enable_coprocessor(void);
	void enable_instruction_cache(void);
	void enable_data_cache(void);
	void disable_coprocessor(void);
	void disable_instruction_cache(void);
	void disable_data_cache(void);
	unsigned int shared_memory_read_wait(void) const
		{ return shmem_ctl.read_wait(); }
	unsigned int shared_memory_write_wait(void) const
		{ return shmem_ctl.write_wait(); }
	unsigned int sync_memory_read_wait(void) const
		{ return shfad_ctl.read_wait(); }
	unsigned int sync_memory_write_wait(void) const
		{ return shfad_ctl.write_wait(); }
	void set_shared_memory_read_wait(unsigned int);
	void set_shared_memory_write_wait(unsigned int);
	void set_sync_memory_read_wait(unsigned int);
	void set_sync_memory_write_wait(unsigned int);
};

#endif /* SHARED_BUS_ARCHITECTURE_H */
