/*
 * <<< hypercube_duato_router.h >>>
 *
 * --- Router class for Hypercube, with Duato's protocol
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef HYPERCUBE_DUATO_ROUTER_H
#define HYPERCUBE_DUATO_ROUTER_H

#include <isis/isis.h>

template <class P>
class hypercube_duato_router : public duato_router<P>
{
private:
	typedef hypercube_duato_router<P> thisclass;
	typedef duato_router<P> inherited;
public:
	typedef inherited::packet_type packet_type;
	typedef inherited::node_address_type node_address_type;
private:
	const static size_t virtual_channel_size = 2;
	const static size_t ADAPTIVE_VR_CHANNEL = 0;
	const static size_t ESCAPE_VR_CHANNEL = 1;
	enum state_type { READY, BUSY };
	size_t dim_, size_;
	size_t adapt_ph_master_;
protected:
	virtual void adaptive_routing(packet_type&, channel_controller&);
	virtual void escape_routing(packet_type&, channel_controller&);
public:
	hypercube_duato_router(void);
	hypercube_duato_router(const hypercube_duato_router&);
	virtual ~hypercube_duato_router() {}
	void set_dimension(size_t);
};

template <class P>
hypercube_duato_router<P>::hypercube_duato_router(void)
	: inherited(),
	  dim_(0),
	  size_(1),
	  adapt_ph_master_(0)
{
	set_input_size(dim_ + 1);
	set_output_size(dim_ + 1);
	set_channel_size(dim_, 1);
	set_buffer_size(dim_, 1);
}

template <class P>
hypercube_duato_router<P>::hypercube_duato_router
	(const hypercube_duato_router<P>& a)
	: inherited(a),
	  dim_(a.dim_),
	  size_(a.size_),
	  adapt_ph_master_(a.adapt_ph_master_)
{}

template <class P>
void hypercube_duato_router<P>::set_dimension(size_t a)
{
	size_t buf_size = buffer_size(0);
	dim_ = a;
	size_ = (1 << a);
	set_input_size(dim_ + 1);
	set_output_size(dim_ + 1);
	for (size_t i = 0; i < dim_; i++) {
		set_channel_size(i, virtual_channel_size);
		set_buffer_size(i, buf_size);
	}
	set_channel_size(dim_, 1);
	set_buffer_size(dim_, buf_size);
	adapt_ph_master_ = 0;
}

template <class P>
void hypercube_duato_router<P>::adaptive_routing
	(hypercube_duato_router<P>::packet_type& pkt,
	 channel_controller& ctl)
{
	if (pkt.destination() == node_address()) return;
	size_t diff = (pkt.destination() ^ node_address());
	for (size_t i = 0; i < dim_; i++) {
		size_t pid = (adapt_ph_master_ + i) % dim_;
		size_t vid = ADAPTIVE_VR_CHANNEL;
		if (((diff >> pid) & 1 != 0) &&
			!is_locked_output_virtual_channel(pid, vid) &&
			!output_channel(pid).full(vid)) {
			ctl.set_destination(pid, vid);
			lock_output_virtual_channel(pid, vid);
			adapt_ph_master_ = (adapt_ph_master_ + i + 1) % dim_;
			return;
		}
	}
}

template <class P>
void hypercube_duato_router<P>::escape_routing
	(hypercube_duato_router<P>::packet_type& pkt,
	 channel_controller& ctl)
{
	if (pkt.destination() == node_address()) {
		// destination is here, transmit to PE
		ctl.set_destination(dim_, 0);
		return;
	}
	size_t diff = (pkt.destination() ^ node_address());
	size_t next_ch;
	for (next_ch = 0; ; next_ch++) {
		if ((diff & 1) != 0) break;
		diff >>= 1;
	}
	assert(next_ch < dim_ + 1);
	ctl.set_destination(next_ch, ESCAPE_VR_CHANNEL);
}

#endif /* HYPERCUBE_DUATO_ROUTER_H */
