/*
 * <<< single_router_test.cc >>>
 *
 * --- Test program for router class, single version
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cassert>
#include <cstdlib>
#include <list>
#include <vector>
#include <iostream>
#include <iomanip>
#include <string>
#include <isis/network_packet_base.h>
#include <isis/network_packet_sender.h>
#include <isis/network_packet_receiver.h>
#include "deterministic_router.h"

#define rnd() ((unsigned int)((seed = 1566083941UL * seed + 1) >> 16))
static unsigned long seed = 1;

// typedefs
class myrouter;
typedef size_t address_type;
typedef network_packet_base<address_type> packet_type;
typedef myrouter router_type;

// 4 input, 4 output router
class myrouter : public deterministic_router<packet_type>
{
private:
	typedef myrouter thisclass;
	typedef deterministic_router<packet_type> inherited;
public:
	typedef inherited::packet_type packet_type;
protected:
	virtual void routing(const packet_type&, channel_controller&);
public:
	myrouter(void);
	myrouter(const myrouter&);
	virtual ~myrouter() {}
};

myrouter::myrouter(void)
	: inherited()
{
	set_input_size(4);
	set_output_size(4);
	for (size_t i = 0; i < 4; i++) {
		set_channel_size(i, 1);
		set_buffer_size(i, 1);
	}
}

myrouter::myrouter(const myrouter& a)
	: inherited(a)
{}

void myrouter::routing(const packet_type& pkt, channel_controller& ctl)
{
	ctl.set_destination(pkt.destination(), 0);
}

int main(int, char** argv)
{
	// set default values
	size_t buf_size = 1, pkt_len = 1, iter_count = 100;
	double access_rate = .1;
	bool verbose_flag = false;

	// read command-line option
	while (*++argv != NULL && **argv == '-') {
		switch (*(*argv + 1)) {
		case 'b':
			buf_size = (size_t)atoi(*argv + 2);
			break;
		case 'l':
			pkt_len = (size_t)atoi(*argv + 2);
			break;
		case 'r':
			access_rate = (double)atof(*argv + 2);
			break;
		case 'n':
			iter_count = (size_t)atoi(*argv + 2);
			break;
		case 'v':
			verbose_flag = true;
			break;
		default:
			break;
		}
	}

	// create router, injectors, receivers and connect each other
	router_type rt;
	vector< network_packet_sender<packet_type> > injector(4);
	vector< network_packet_receiver<packet_type> > receiver(4);
	{
		rt.set_node_address(0);
		size_t i;
		for (i = 0; i < injector.size(); i++) {
			rt.set_channel_size(i, 1);
			rt.set_buffer_size(i, buf_size);
			rt.input_channel(i).connect(injector[i].channel());
		}
		for (i = 0; i < receiver.size(); i++) {
			rt.output_channel(i).connect(receiver[i].channel());
		}
	}
	rt.setup();

	// create injection-queue, diag-counters
	vector< list<packet_type*> > inj_que(injector.size());
	size_t sent_count = 0, received_count = 0;

	// show parameters
	cout << "--- test condition ---" << endl
		 << "input links:      " << injector.size() << endl
		 << "output links:     " << receiver.size() << endl
		 << "buffer size:      " << buf_size << endl
		 << "packet length:    " << pkt_len << endl
		 << "access rate:      " << access_rate << endl
		 << "iterations:       " << iter_count << endl << endl;

	// main loop
	for (size_t iter = 0; ; iter++) {
		size_t i;
		// show current status if verbose
		if (verbose_flag) {
			cout << "--- clk:" << iter << " ---" << endl;
			cout << "inj_que:" << endl;
			for (i = 0; i < inj_que.size(); i++) {
				if (!inj_que[i].empty()) {
					cout << ' ' << setw(2) << i << ":[";
					list<packet_type*>::const_iterator p = inj_que[i].begin();
					while (p != inj_que[i].end()) {
						packet_type* pkt = *p;
						p++;
						if (p != inj_que[i].end()) {
							if (pkt == *p) {
								cout << ':';
							} else {
								cout << *pkt << ',';
							}
						} else {
							cout << *pkt;
						}
					}
					cout << ']' << endl;
				}
			}
			cout << "router:" << endl
				 << "  " << rt << endl;
			cout << endl;
		}
		if (iter >= iter_count) break;
		// clock_in: router's clock_in
		rt.clock_in();
		// clock_in: trans inj_que to injector
		for (i = 0; i < inj_que.size(); i++) {
			if (!inj_que[i].empty() && !injector[i].full()) {
				injector[i].put(inj_que[i].back());
				inj_que[i].pop_back();
			}
		}
		// clock_out: router's clock_out
		rt.clock_out();
		// clock_out: injector's clock
		for (i = 0; i < injector.size(); i++) {
			injector[i].clock();
		}
		// clock_out: receiver's clock
		for (i = 0; i < receiver.size(); i++) {
			receiver[i].clock();
			if (receiver[i].full()) {
				packet_type* pkt = receiver[i].get();
				assert(pkt->destination() == i);
				delete pkt;
				received_count++;
			}
		}
		// generate packet and inject to inj_que
		for (i = 0; i < inj_que.size(); i++) {
			if (inj_que[i].size() < 2 &&
				(double(rnd() & 0x7fff) / 0x8000) < access_rate / pkt_len) {
				packet_type* p = new packet_type;
				p->set_source(i);
				p->set_destination(rnd() % receiver.size());
				p->set_length(pkt_len);
				inj_que[i].push_front(p);
				sent_count++;
			}
		}
	}

	// show results
	cout << "--- results ---" << endl
		 << "sent:       " << setw(8) << sent_count << endl
		 << "received:   " << setw(8) << received_count << endl
		 << "throughput: " << setw(8)
		 << (double(received_count) * pkt_len / iter_count) << endl;

	// exit
	return 0;
}
