/*
 * <<< router.h >>>
 *
 * --- Router class 'router'
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef ROUTER_H
#define ROUTER_H

#include <algorithm>
#include <iostream>
#include <vector>
#include <isis/synchronous_unit.h>
#include <isis/virtual_channel.h>
#include <isis/crossbar.h>

template <class P>
class router : public synchronous_unit
{
private:
	typedef router<P> thisclass;
	typedef synchronous_unit inherited;
public:
	typedef P packet_type;
	typedef typename P::address_type node_address_type;
protected:
	class channel_controller
	{
	private:
		typedef channel_controller thisclass;
		typedef pair<size_t, size_t> node_address_type;
		typedef vector<node_address_type> node_address_container_type;
		enum state_type { READY, WAIT, BUSY };
		size_t ph_id_, vr_id_;
		size_t dst_ph_size_, dst_vr_size_;
		state_type state_;
		node_address_container_type dst_tbl_;
		size_t dst_num_;
		size_t remain_count_;
		bool activate_flag_;
	public:
		inline channel_controller(void);
		inline channel_controller(const thisclass&);
		~channel_controller() {}
		// setup
		void set_id(size_t i, size_t j) { ph_id_ = i;  vr_id_ = j; }
		inline void set_destination_size(size_t, size_t);
		// query my id
		size_t physical_channel_id(void) const { return ph_id_; }
		size_t virtual_channel_id(void) const { return vr_id_; }
		// routing functions
		void set_destination(size_t i, size_t j)
			{ dst_tbl_[dst_num_++] = node_address_type(i, j); }
		size_t destination_number(void) const { return dst_num_; }
		node_address_type destination(size_t i = 0) const
			{ return dst_tbl_[i]; }
		void clear_destination_table(void) { dst_num_ = 0; }
		// state transition
		bool is_ready(void) const { return state_ == READY; }
		bool is_wait(void) const { return state_ == WAIT; }
		bool is_busy(void) const { return state_ == BUSY; }
		bool is_activated(void) const { return activate_flag_; }
		void become_ready(void) { state_ = READY; }
		void become_wait(void) { state_ = WAIT; }
		void become_busy(void) { state_ = BUSY; }
		void activate(void) { activate_flag_ = true; }
		void inactivate(void) { activate_flag_ = false; }
		void clock(void)
			{ activate_flag_ = false;
			  if (--remain_count_ == 0) dst_num_ = 0, state_ = READY; }
		void reset(void)
			{ state_ = READY, dst_num_ = 0, remain_count_ = 0; }
		// packet length
		void set_packet_length(size_t a) { remain_count_ = a; }
		size_t remainder_length(void) const { return remain_count_; }
	};
private:
	vector<virtual_channel_input> in_ch_;
	vector<virtual_channel_output> out_ch_;
	vector<channel_controller> ctl_tbl_;
	vector<bool> outvr_lock_tbl_;
	crossbar cb_;
	size_t in_vr_size_, out_vr_size_;
	node_address_type adr_;
	size_t total_hop_count_;
protected:
	// internal resources
	const channel_controller& internal_controller(size_t i, size_t j) const
		{ return ctl_tbl_[i * in_vr_size_ + j]; }
	channel_controller& internal_controller(size_t i, size_t j)
		{ return ctl_tbl_[i * in_vr_size_ + j]; }
	const crossbar& internal_crossbar(void) const { return cb_; }
	crossbar& internal_crossbar(void) { return cb_; }
	bool is_locked_output_virtual_channel(size_t i, size_t j) const
		{ return outvr_lock_tbl_[i * out_vr_size_ + j]; }
	void lock_output_virtual_channel(size_t i, size_t j)
		{ outvr_lock_tbl_[i * out_vr_size_ + j] = true; }
	void unlock_output_virtual_channel(size_t i, size_t j)
		{ outvr_lock_tbl_[i * out_vr_size_ + j] = false; }
	void set_input_size(size_t);
	void set_output_size(size_t);
	// diagnostics
	void increment_total_hop_count(void) { total_hop_count_++; }
public:
	router(void);
	router(const thisclass&);
	virtual ~router();
	// state transition functions
	virtual void setup(void);
	virtual void reset(void);
	virtual void reset_counter(void);
	// output to ostream
	virtual void output(ostream&) const;
	// channel management
	const virtual_channel_input& input_channel(size_t i) const
		{ return in_ch_[i]; }
	const virtual_channel_output& output_channel(size_t i) const
		{ return out_ch_[i]; }
	virtual_channel_input& input_channel(size_t i) { return in_ch_[i]; }
	virtual_channel_output& output_channel(size_t i) { return out_ch_[i]; }
	// address management
	node_address_type node_address(void) const { return adr_; }
	void set_node_address(node_address_type a) { adr_ = a; }
	// diagnostics
	size_t total_hop_count(void) const { return total_hop_count_; }
	// parameter settings
	size_t input_size(void) const { return in_ch_.size(); }
	size_t output_size(void) const { return out_ch_.size(); }
	size_t channel_size(size_t i) const { return in_ch_[i].channel_size(); }
	size_t buffer_size(size_t i) const { return in_ch_[i].buffer_size(); }
	size_t buffer_length(size_t i, size_t j) const
		{ return in_ch_[i].length(j); }
	void set_channel_size(size_t, size_t);
	void set_buffer_size(size_t, size_t);
};

template <class P>
inline router<P>::channel_controller::channel_controller(void)
	: ph_id_(0),
	  vr_id_(0),
	  dst_ph_size_(0),
	  dst_vr_size_(0),
	  state_(READY),
	  dst_tbl_(0),
	  dst_num_(0)
{}

template <class P>
inline router<P>::channel_controller::channel_controller
	(const router<P>::channel_controller& a)
	: ph_id_(a.ph_id_),
	  vr_id_(a.vr_id_),
	  dst_ph_size_(a.dst_ph_size_),
	  dst_vr_size_(a.dst_vr_size_),
	  state_(a.state_),
	  dst_tbl_(a.dst_tbl_),
	  dst_num_(a.dst_num_),
	  remain_count_(a.remain_count_)
{}

template <class P>
inline void router<P>::channel_controller::set_destination_size
	(size_t i, size_t j)
{
	dst_tbl_.resize(i * j);
	dst_ph_size_ = i;
	dst_vr_size_ = j;
	dst_num_ = 0;
}

template <class P>
router<P>::router(void)
	: in_ch_(0),
	  out_ch_(0),
	  ctl_tbl_(0),
	  outvr_lock_tbl_(0),
	  cb_(),
	  in_vr_size_(0),
	  out_vr_size_(0),
	  adr_(),
	  total_hop_count_(0)
{}

template <class P>
router<P>::router(const router<P>& a)
	: in_ch_(a.in_ch_),
	  out_ch_(a.out_ch_),
	  ctl_tbl_(a.ctl_tbl_),
	  outvr_lock_tbl_(a.outvr_lock_tbl_),
	  cb_(a.cb_),
	  in_vr_size_(a.in_vr_size_),
	  out_vr_size_(a.out_vr_size_),
	  adr_(a.adr_),
	  total_hop_count_(a.total_hop_count_)
{}

template <class P>
router<P>::~router()
{}

template <class P>
void router<P>::set_input_size(size_t a)
{
	in_ch_.resize(a);
}

template <class P>
void router<P>::set_output_size(size_t a)
{
	out_ch_.resize(a);
}

template <class P>
void router<P>::set_channel_size(size_t i, size_t a)
{
	in_ch_[i].set_channel_size(a);
}

template <class P>
void router<P>::set_buffer_size(size_t i, size_t a)
{
	in_ch_[i].set_buffer_size(a);
}

template <class P>
void router<P>::setup(void)
{
	const size_t in_ph_size = input_size(), out_ph_size = output_size();
	size_t in_vr_size = 0, out_vr_size = 0;
	size_t i;
	for (i = 0; i < in_ph_size; i++) {
		size_t tmp = in_ch_[i].channel_size();
		if (tmp > in_vr_size) in_vr_size = tmp;
	}
	for (i = 0; i < out_ph_size; i++) {
		size_t tmp = out_ch_[i].channel_size();
		if (tmp > out_vr_size) out_vr_size = tmp;
	}
	in_vr_size_ = in_vr_size;
	out_vr_size_ = out_vr_size;
	ctl_tbl_.resize(in_ph_size * in_vr_size);
	for (i = 0; i < in_ph_size; i++) {
		for (size_t j = 0; j < input_channel(i).channel_size(); j++) {
			channel_controller& ctl(internal_controller(i, j));
			ctl.set_id(i, j);
			ctl.set_destination_size(out_ph_size, out_vr_size);
		}
	}
	outvr_lock_tbl_.resize(out_ph_size * out_vr_size);
	cb_.resize(in_ph_size, out_ph_size);
	thisclass::reset();
}

template <class P>
void router<P>::reset(void)
{
	size_t i;
	for (i = 0; i < in_ch_.size(); i++) in_ch_[i].clear();
	for (i = 0; i < out_ch_.size(); i++) out_ch_[i].clear();
	for (i = 0; i < ctl_tbl_.size(); i++) ctl_tbl_[i].reset();
	fill(outvr_lock_tbl_.begin(), outvr_lock_tbl_.end(), false);
	cb_.clear();
}

template <class P>
void router<P>::reset_counter(void)
{
	total_hop_count_ = 0;
}

template <class P>
void router<P>::output(ostream& os) const
{
	size_t i;
	for (i = 0; i < input_size(); i++) {
		os << "ch" << i << ':' << input_channel(i);
		if (i < input_size() - 1) os << ' ';
	}
}

#endif /* ROUTER_H */
