/* -*- C++ -*-
 *
 * <<< message_passing_network_interface.h >>>
 *
 * --- Network interface unit for message-passing class
 *     'message_passing_network_interface'
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef MESSAGE_PASSING_NETWORK_INTERFACE_H
#define MESSAGE_PASSING_NETWORK_INTERFACE_H 1

#include <iostream>
#include <isis/bus_port.h>
#include <isis/mapped_memory.h>
#include <isis/io_protocol.h>
#include <isis/network_interface.h>

// #define MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG

template <class P, class A, class D = A, size_t Dsz = sizeof(D),
		  class Ch = char>
class message_passing_network_interface : public network_interface<P>
{
private:
	typedef message_passing_network_interface<P, A, D, Dsz, Ch> thisclass;
	typedef network_interface<P> inherited;
public:
	typedef inherited::packet_type packet_type;
	typedef inherited::node_address_type node_address_type;
	typedef A address_type;
	typedef D data_type;
	typedef Ch char_type;
	typedef A size_type;
	typedef bus_port<A, D> bus_port_type;
	typedef mapped_memory<A, D, Dsz, Ch> memory_type;
	typedef typename bus_port_type::transaction_id_type transaction_id_type;
	static const size_t sizeof_data_type = Dsz;
private:
	enum state_type {
		READY,
		REG_SINGLE_READ_WAIT,
		REG_SINGLE_READ_ACK,
		REG_SINGLE_READ_TRANS,
		REG_SINGLE_WRITE_WAIT,
		REG_SINGLE_WRITE_ACK,
		REG_SINGLE_WRITE_TRANS,
		REG_MULTI_READ_WAIT,
		REG_MULTI_READ_ACK,
		REG_MULTI_READ_TRANS,
		REG_MULTI_WRITE_WAIT,
		REG_MULTI_WRITE_ACK,
		REG_MULTI_WRITE_TRANS,
		DMA_READ_REQ_OWNER,
		DMA_READ_SEND,
		DMA_READ_WAIT,
		DMA_READ_TRANS,
		DMA_READ_END,
		DMA_WRITE_REQ_OWNER,
		DMA_WRITE_SEND,
		DMA_WRITE_WAIT,
		DMA_WRITE_TRANS,
		DMA_WRITE_END
	};
	const static size_t default_max_packet_length = 257; // in flits
	const static size_t hardware_register_size = 16; // in words
	state_type state_;
	bool req_wait_flag_;
	bus_port_type bus_if_;
	memory_type mem_buf_;
	unsigned int mem_read_wait_, mem_write_wait_;
	size_t dma_trans_size_, max_packet_length_, max_data_size_;
	unsigned int wait_count;
	size_t total_packet_count, packet_number;
	size_t total_cluster_count, cluster_number;
	address_type base_address, address;
	packet_type* curr_pkt;
protected:
	virtual packet_type* generate_packet(node_address_type,
										 node_address_type, size_t);
public:
	message_passing_network_interface(void);
	message_passing_network_interface(const thisclass&);
	virtual ~message_passing_network_interface();
	// state transition
	inline virtual void clock_in(void);
	inline virtual void clock_out(void);
	virtual void reset(void);
	// bus port
	const bus_port_type& bus_port_ref(void) const { return bus_if_; }
	bus_port_type& bus_port_ref(void) { return bus_if_; }
	// inner memory
	unsigned int memory_read_wait(void) const { return mem_read_wait_; }
	unsigned int memory_write_wait(void) const { return mem_write_wait_; }
	void set_memory_read_wait(unsigned int);
	void set_memory_write_wait(unsigned int);
	const memory_type& memory(void) const { return mem_buf_; }
	memory_type& memory(void) { return mem_buf_; }
	void set_memory_address(address_type a) { mem_buf_.set_top(a); }
	// misc parameters (DMA transfer)
	size_t dma_transfer_size(void) const { return dma_trans_size_; }
	void set_dma_transfer_size(size_t a) const { dma_trans_size_ = a; }
	size_t max_packet_length(void) const { return max_packet_length_; }
	void set_max_packet_length(size_t a)
	{ max_packet_length_ = a - ((a - 1) % dma_trans_size_);
	  max_data_size_ = (max_packet_length_ - 1) * sizeof_data_type; }
};

template <class P, class A, class D, size_t Dsz, class Ch>
message_passing_network_interface<P, A, D, Dsz, Ch>::
	message_passing_network_interface(void)
	: inherited(),
	  state_(READY),
	  req_wait_flag_(false),
	  bus_if_(),
	  mem_buf_(address_type(0), sizeof_data_type * hardware_register_size),
	  mem_read_wait_(0),
	  mem_write_wait_(0),
	  dma_trans_size_(4),
	  max_packet_length_(default_max_packet_length),
	  max_data_size_((max_packet_length_ - 1) * sizeof_data_type),
	  curr_pkt(NULL)
{}

template <class P, class A, class D, size_t Dsz, class Ch>
message_passing_network_interface<P, A, D, Dsz, Ch>::
	message_passing_network_interface
	(const message_passing_network_interface<P, A, D, Dsz, Ch>& a)
	: inherited(a),
	  state_(a.state_),
	  req_wait_flag_(a.req_wait_flag_),
	  bus_if_(),
	  mem_buf_(a.mem_buf_),
	  mem_read_wait_(a.mem_read_wait_),
	  mem_write_wait_(a.mem_write_wait_),
	  dma_trans_size_(a.dma_trans_size_),
	  max_packet_length_(a.max_packet_length_),
	  max_data_size_(a.max_data_size_),
	  curr_pkt(NULL)
{}

template <class P, class A, class D, size_t Dsz, class Ch>
message_passing_network_interface<P, A, D, Dsz, Ch>::
	~message_passing_network_interface()
{
	if (curr_pkt != NULL) delete curr_pkt;
}

template <class P, class A, class D, size_t Dsz, class Ch>
message_passing_network_interface<P, A, D, Dsz, Ch>::packet_type*
	message_passing_network_interface<P, A, D, Dsz, Ch>::generate_packet
	(message_passing_network_interface<P, A, D, Dsz, Ch>::node_address_type src,
	 message_passing_network_interface<P, A, D, Dsz, Ch>::node_address_type dst,
	 size_t dat_sz)
{
	packet_type* p = new packet_type;
	p->set_source(src);
	p->set_destination(dst);
	p->set_length(dat_sz + 1);
	p->set_data_size(dat_sz);
	return p;
}

template <class P, class A, class D, size_t Dsz, class Ch>
inline void message_passing_network_interface<P, A, D, Dsz, Ch>::clock_in(void)
{
	if (!bus_if_.is_connected()) return;
	if (wait_count > 0) wait_count--;
	switch (state_) {
	case READY:
		if (bus_if_.is_ready()) {
			if (req_wait_flag_) {
				int access_type = int(mem_buf_.read(mem_buf_.top()));
				switch (access_type) {
				case MPNI_IO_GET_PARAMETER:
					{
						data_type d1, d2, d3;
						d1 = data_type(max_data_size_);
						d2 = data_type(send_queue_size());
						d3 = data_type(receive_queue_size());
						mem_buf_.write(mem_buf_.top() + sizeof_data_type, d1);
						mem_buf_.write(mem_buf_.top() + sizeof_data_type * 2,
									   d2);
						mem_buf_.write(mem_buf_.top() + sizeof_data_type * 3,
									   d3);
						mem_buf_.write(mem_buf_.top(),
									   data_type(MPNI_IO_READY));
						req_wait_flag_ = false;
					}
					break;
				case MPNI_IO_SENSE_SEND_QUEUE:
					{
						data_type d1;
						d1 = data_type(send_queue_size() - send_queue_length());
						mem_buf_.write(mem_buf_.top() + sizeof_data_type, d1);
#ifdef MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						long flags = cerr.flags();
						cerr << hex
							 << "message_passing_network_interface(0x"
							 << node_address() << "): sense_send_queue(): "
							 << dec;
						if (d1 > 0) {
							cerr << "send queue has " << d1 << " space(s)";
						} else {
							cerr << "send queue is full";
						}
						cerr << endl;
						cerr.flags(flags);
#endif // MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						mem_buf_.write(mem_buf_.top(),
									   data_type(MPNI_IO_READY));
						req_wait_flag_ = false;
					}
					break;
				case MPNI_IO_SENSE_RECEIVE_QUEUE:
					{
						data_type d1, d2, d3;
						d1 = d2 = data_type(0); // for ignore compiler warnings
						d3 = data_type(receive_queue_length());
						mem_buf_.write(mem_buf_.top() + sizeof_data_type * 3,
									   d3);
						if (d3 > 0) {
							d1 = data_type(look_received_packet()->source());
							d2 = data_type(
									(look_received_packet()->length() - 1)
									* sizeof_data_type
								 );
							mem_buf_.write(mem_buf_.top()
										   + sizeof_data_type, d1);
							mem_buf_.write(mem_buf_.top()
										   + sizeof_data_type * 2, d2);
						}
#ifdef MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						long flags = cerr.flags();
						cerr << hex
							 << "message_passing_network_interface(0x"
							 << node_address() << "): sense_receive_queue(): "
							 << dec;
						if (d3 > 0) {
							cerr << "receive queue has "
								 << d3 << " packet(s), source of front is "
								 << d1;
						} else {
							cerr << "receive queue is empty";
						}
						cerr << endl;
						cerr.flags(flags);
#endif // MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						mem_buf_.write(mem_buf_.top(),
									   data_type(MPNI_IO_READY));
						req_wait_flag_ = false;
					}
					break;
				case MPNI_IO_SEND:
					assert(is_ready_to_enqueue());
					{
						// destination, base address, size(in byte)
						data_type d1, d2, d3;
						d1 = mem_buf_.read(mem_buf_.top()
										   + sizeof_data_type);
						d2 = mem_buf_.read(mem_buf_.top()
										   + sizeof_data_type * 2);
						d3 = mem_buf_.read(mem_buf_.top()
										   + sizeof_data_type * 3);
						if (size_t(d3) > max_data_size_) {
							d3 = data_type(max_data_size_);
						}
#ifdef MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						long flags = cerr.flags();
						cerr << hex
							 << "message_passing_network_interface(0x"
							 << node_address() << "): send(0x"
							 << d1 << ", 0x" << d2 << ", 0x" << d3 << ')'
							 << endl;
						cerr.flags(flags);
#endif // MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						base_address = d2;
						total_cluster_count
							= d3 / (dma_trans_size_ * sizeof_data_type);
						cluster_number = 0;
						size_t sz = d3 / dma_trans_size_;
						curr_pkt = generate_packet(node_address(),
												   node_address_type(d1), sz);
						if (sz > 0) {
							state_ = DMA_READ_REQ_OWNER;
						} else {
							enqueue_send_packet(curr_pkt);
							curr_pkt = NULL;
							mem_buf_.write(mem_buf_.top(),
										   data_type(MPNI_IO_READY));
							req_wait_flag_ = false;
							state_ = READY;
						}
					}
					break;
				case MPNI_IO_RECEIVE:
					assert(is_ready_to_dequeue());
					{
						// base address, size(in byte)
						data_type d1, d2;
						d1 = mem_buf_.read(mem_buf_.top()
										   + sizeof_data_type);
						d2 = mem_buf_.read(mem_buf_.top()
										   + sizeof_data_type * 2);
						if (size_t(d2) > max_data_size_) {
							d2 = data_type(max_data_size_);
						}
#ifdef MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						long flags = cerr.flags();
						cerr << hex
							 << "message_passing_network_interface(0x"
							 << node_address() << "): receive(0x"
							 << d1 << ", 0x" << d2 << ')' << endl;
						cerr.flags(flags);
#endif // MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						base_address = d1;
						total_cluster_count
							= d2 / (dma_trans_size_ * sizeof_data_type);
						cluster_number = 0;
						size_t sz = d2 / dma_trans_size_;
						if (sz > 0) {
							curr_pkt = dequeue_received_packet();
							state_ = DMA_WRITE_REQ_OWNER;
#ifdef MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
							const char fill = cerr.fill();
							const long flags = cerr.flags();
							cerr << hex
								 << "message_passing_network_interface(0x"
								 << node_address()
								 << "): received packet is: ("
								 << setfill('0');
							for (size_t i = 0; i < curr_pkt->data_size(); i++) {
								cerr << setw(8) << curr_pkt->data(i);
								if (i < curr_pkt->data_size() - 1) cerr << ',';
							}
							cerr << ')' << endl;
							cerr.fill(fill);
							cerr.flags(flags);
#endif // MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						} else {
							delete dequeue_received_packet();
							mem_buf_.write(mem_buf_.top(),
										   data_type(MPNI_IO_READY));
							req_wait_flag_ = false;
							state_ = READY;
						}
					}
					break;
				case MPNI_IO_DELETE:
					assert(is_ready_to_dequeue());
					{
						delete dequeue_received_packet();
#ifdef MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						long flags = cerr.flags();
						cerr << hex
							 << "message_passing_network_interface(0x"
							 << node_address() << "): delete(): "
							 << "1 packet deleted" << endl;
						cerr.flags(flags);
#endif // MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
						mem_buf_.write(mem_buf_.top(),
									   data_type(MPNI_IO_READY));
						req_wait_flag_ = false;
					}
					break;
				default:
					// protocol error
					assert(false);
					break;
				}
			}
			break;
		}
		if (mem_buf_.is_valid(bus_if_.address())) {
			address = bus_if_.address();
			if (bus_if_.is_single_read_request()) {
				wait_count = memory_read_wait();
				state_ = (wait_count > 1) ?
							 REG_SINGLE_READ_WAIT : REG_SINGLE_READ_ACK;
			} else if (bus_if_.is_single_write_request()) {
				assert(!req_wait_flag_);
				wait_count = memory_write_wait();
				state_ = (wait_count > 1) ?
							 REG_SINGLE_WRITE_WAIT : REG_SINGLE_WRITE_ACK;
			} else if (bus_if_.is_multi_read_request()) {
				base_address = address;
				total_packet_count = bus_if_.total_packet_count();
				packet_number = 0;
				wait_count = memory_read_wait();
				state_ = (wait_count > 1) ?
							 REG_MULTI_READ_WAIT : REG_MULTI_READ_ACK;
			} else if (bus_if_.is_multi_write_request()) {
				assert(!req_wait_flag_);
				base_address = address;
				total_packet_count = bus_if_.total_packet_count();
				packet_number = 0;
				wait_count = memory_write_wait();
				state_ = (wait_count > 1) ?
							 REG_MULTI_WRITE_WAIT : REG_MULTI_WRITE_ACK;
			}
		}
		break;
	case REG_SINGLE_READ_WAIT:
		if (wait_count <= 1) state_ = REG_SINGLE_READ_ACK;
		break;
	case REG_SINGLE_WRITE_WAIT:
		if (wait_count <= 1) state_ = REG_SINGLE_WRITE_ACK;
		break;
	case REG_SINGLE_WRITE_TRANS:
		if (bus_if_.is_single_write_data()) {
			mem_buf_.write(address, bus_if_.data());
			if (address == mem_buf_.top()) req_wait_flag_ = true;
			state_ = READY;
		}
		break;
	case REG_MULTI_READ_WAIT:
		if (wait_count <= 1) state_ = REG_MULTI_READ_ACK;
		break;
	case REG_MULTI_WRITE_WAIT:
		if (wait_count <= 1) state_ = REG_MULTI_WRITE_ACK;
		break;
	case REG_MULTI_WRITE_TRANS:
		if (bus_if_.is_multi_write_data()) {
			mem_buf_.write(address, bus_if_.data());
			if (address == mem_buf_.top()) req_wait_flag_ = true;
			if (++packet_number < total_packet_count) {
				address += sizeof_data_type;
			} else {
				state_ = READY;
			}
		}
		break;
	case DMA_READ_REQ_OWNER:
		if (bus_if_.is_owner()) state_ = DMA_READ_SEND;
		break;
	case DMA_READ_WAIT:
		if (bus_if_.is_multi_read_ack()) {
			packet_number = 0;
			state_ = DMA_READ_TRANS;
		}
		break;
	case DMA_READ_TRANS:
		if (bus_if_.is_multi_read_data()) {
			curr_pkt->set_data(cluster_number * dma_trans_size_ + packet_number,
							   bus_if_.data());
			if (++packet_number == dma_trans_size_) {
				if (++cluster_number < total_cluster_count) {
					packet_number = 0;
					state_ = DMA_READ_SEND;
				} else {
					enqueue_send_packet(curr_pkt);
#ifdef MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
					char fill = cerr.fill();
					long flags = cerr.flags();
					cerr << hex
						 << "message_passing_network_interface(0x"
						 << node_address() << "): sent packet is: ("
						 << setfill('0');
					for (size_t i = 0; i < curr_pkt->data_size(); i++) {
						cerr << setw(8) << curr_pkt->data(i);
						if (i < curr_pkt->data_size() - 1) cerr << ',';
					}
					cerr << ')' << endl;
					cerr.fill(fill);
					cerr.flags(flags);
#endif // MESSAGE_PASSING_NETWORK_INTERFACE_DEBUG
					curr_pkt = NULL;
					mem_buf_.write(mem_buf_.top(), data_type(MPNI_IO_READY));
					req_wait_flag_ = false;
					state_ = DMA_READ_END;
				}
			}
		}
		break;
	case DMA_WRITE_REQ_OWNER:
		if (bus_if_.is_owner()) state_ = DMA_WRITE_SEND;
		break;
	case DMA_WRITE_WAIT:
		if (bus_if_.is_multi_write_ack()) {
			packet_number = 0;
			state_ = DMA_WRITE_TRANS;
		}
		break;
	default:
		break;
	}
	inherited::clock_in();
}

template <class P, class A, class D, size_t Dsz, class Ch>
inline void message_passing_network_interface<P, A, D, Dsz, Ch>::clock_out(void)
{
	inherited::clock_out();
	if (!bus_if_.is_connected()) return;
	switch (state_) {
	case REG_SINGLE_READ_ACK:
		bus_if_.send_single_read_ack();
		state_ = REG_SINGLE_READ_TRANS;
		break;
	case REG_SINGLE_READ_TRANS:
		bus_if_.send_single_read_data(mem_buf_.read(address));
		state_ = READY;
		break;
	case REG_SINGLE_WRITE_ACK:
		bus_if_.send_single_write_ack();
		state_ = REG_SINGLE_WRITE_TRANS;
		break;
	case REG_MULTI_READ_ACK:
		bus_if_.send_multi_read_ack();
		state_ = REG_MULTI_READ_TRANS;
		break;
	case REG_MULTI_READ_TRANS:
		bus_if_.send_multi_read_data(mem_buf_.read(address),
									 total_packet_count, packet_number);
		if (++packet_number < total_packet_count) {
			address += sizeof_data_type;
		} else {
			state_ = READY;
		}
		break;
	case REG_MULTI_WRITE_ACK:
		bus_if_.send_multi_write_ack();
		state_ = REG_MULTI_WRITE_TRANS;
		break;
	case DMA_READ_REQ_OWNER:
		bus_if_.request_ownership();
		break;
	case DMA_READ_SEND:
		bus_if_.send_multi_read_request
			(base_address + cluster_number * dma_trans_size_ * sizeof_data_type,
			 dma_trans_size_);
		state_ = DMA_READ_WAIT;
		break;
	case DMA_READ_END:
		bus_if_.clear();
		bus_if_.release_ownership();
		state_ = READY;
		break;
	case DMA_WRITE_REQ_OWNER:
		bus_if_.request_ownership();
		break;
	case DMA_WRITE_SEND:
		bus_if_.send_multi_write_request
			(base_address + cluster_number * dma_trans_size_ * sizeof_data_type,
			 dma_trans_size_);
		state_ = DMA_WRITE_WAIT;
		break;
	case DMA_WRITE_TRANS:
		{
			size_t idx = cluster_number * dma_trans_size_ + packet_number;
			data_type v = (idx < curr_pkt->data_size())
							? data_type(curr_pkt->data(idx)) : data_type(0);
			bus_if_.send_multi_write_data(v, dma_trans_size_, packet_number);
			if (++packet_number == dma_trans_size_) {
				if (++cluster_number < total_cluster_count) {
					packet_number = 0;
					state_ = DMA_WRITE_SEND;
				} else {
					delete curr_pkt;
					curr_pkt = NULL;
					mem_buf_.write(mem_buf_.top(), data_type(MPNI_IO_READY));
					req_wait_flag_ = false;
					state_ = DMA_WRITE_END;
				}
			}
		}
		break;
	case DMA_WRITE_END:
		bus_if_.clear();
		bus_if_.release_ownership();
		state_ = READY;
		break;
	default:
		break;
	}
}

template <class P, class A, class D, size_t Dsz, class Ch>
void message_passing_network_interface<P, A, D, Dsz, Ch>::reset(void)
{
	inherited::reset();
	state_ = READY;
}

template <class P, class A, class D, size_t Dsz, class Ch>
void message_passing_network_interface<P, A, D, Dsz, Ch>::set_memory_read_wait
	(unsigned int a)
{
	mem_read_wait_ = a;
}

template <class P, class A, class D, size_t Dsz, class Ch>
void message_passing_network_interface<P, A, D, Dsz, Ch>::set_memory_write_wait
	(unsigned int a)
{
	mem_write_wait_ = a;
}

#endif /* MESSAGE_PASSING_NETWORK_INTERFACE_H */
