/* -*- C++ -*-
 *
 * <<< generic_processing_element.h >>>
 *
 * --- Abstract generic processing element class 'generic_processing_element'
 *     Copyright (C) 1997-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef GENERIC_PROCESSING_ELEMENT_H
#define GENERIC_PROCESSING_ELEMENT_H 1

#include <isis/single_processing_element.h>
#include <isis/multi_io_unit.h>
#include <isis/bus_error_detector.h>
#include <isis/memory_control_unit.h>
#include <isis/memory_map.h>

template <class P, class A = typename P::address_type,
		  class D = typename P::data_type,
		  size_t Dsz = sizeof(D), class Ch = char>
class generic_processing_element
	: public single_processing_element<A, D, Dsz, Ch>
{
private:
	typedef generic_processing_element<P, A, D, Dsz, Ch> thisclass;
	typedef single_processing_element<A, D, Dsz, Ch> inherited;
public:
	typedef typename inherited::address_type address_type;
	typedef typename inherited::data_type data_type;
	typedef typename inherited::char_type char_type;
	typedef typename inherited::size_type size_type;
	typedef P processor_type;
	typedef typename inherited::memory_map_type memory_map_type;
	static const size_type sizeof_data_type = inherited::sizeof_data_type;
protected:
	processor_type pu;
private:
	memory_map_type mem_map;
	memory_control_unit<A, D, Dsz, Ch> mem_ctl;
	multi_io_unit<A, D, Dsz, Ch> multi_io;
	bus_error_detector<A, D> buserror_mon;
	mapped_memory<A, D, Dsz, Ch> local_mem;
public:
	generic_processing_element(void);
	generic_processing_element(const thisclass&);
	virtual ~generic_processing_element();
	virtual const processor_type& processor(void) const { return pu; }
	virtual processor_type& processor(void) { return pu; }
	virtual const memory_map_type& memory_map(void) const { return mem_map; }
	virtual memory_map_type& memory_map(void) { return mem_map; }
	virtual bool is_halt(void) const { return multi_io.is_halt(); }
	virtual bool is_bus_error(void) const
		{ return buserror_mon.is_bus_error(); }
	virtual address_type bus_error_address(void) const
		{ return buserror_mon.bus_error_address(); }
	virtual size_t processor_number(void) const
		{ return multi_io.get_integer_sysinfo("processor_number"); }
	virtual size_t processor_id(void) const
		{ return multi_io.get_integer_sysinfo("processor_id"); }
	virtual void set_processor_number(size_t);
	virtual void set_processor_id(size_t);
	virtual void set_file_table_size(size_t);
	virtual void set_standard_input_stream(istream&);
	virtual void set_standard_output_stream(ostream&);
	virtual void set_standard_error_stream(ostream&);
	virtual void set_commandline_argument(const char* const*);
	virtual int commandline_status(void) const;
	virtual bool is_defined_sysinfo(const string& a) const
		{ return multi_io.is_defined_sysinfo(a); }
	virtual bool is_integer_sysinfo(const string& a) const
		{ return multi_io.is_integer_sysinfo(a); }
	virtual bool is_string_sysinfo(const string& a) const
		{ return multi_io.is_string_sysinfo(a); }
	virtual data_type get_integer_sysinfo(const string& a) const
		{ return multi_io.get_integer_sysinfo(a); }
	virtual const string& get_string_sysinfo(const string& a) const
		{ return multi_io.get_string_sysinfo(a); }
	virtual void set_sysinfo(const string&, data_type);
	virtual void set_sysinfo(const string&, const string&);
	virtual void unset_sysinfo(const string&);
	virtual data_type timer_clock_value(void) const
		{ return multi_io.timer_clock_value(); }
	virtual void set_local_memory_area(address_type, size_type);
	unsigned int local_memory_read_wait(void) const
		{ return mem_ctl.read_wait(); }
	unsigned int local_memory_write_wait(void) const
		{ return mem_ctl.write_wait(); }
	void set_local_memory_read_wait(unsigned int);
	void set_local_memory_write_wait(unsigned int);
	void enable_split_transaction(void);
	void disable_split_transaction(void);
	void set_multi_io_area(address_type, size_type);
	void set_bus_error_timeout(unsigned int);
};

template <class P, class A, class D, size_t Dsz, class Ch>
generic_processing_element<P, A, D, Dsz, Ch>::generic_processing_element(void)
{
	insert_synchronous_unit(pu);
	insert_synchronous_unit(mem_ctl);
	insert_synchronous_unit(multi_io);
	insert_synchronous_unit(buserror_mon);
	pu.port_ref().connect(mem_ctl.port_ref());
	pu.port_ref().connect(multi_io.port_ref());
	pu.port_ref().connect(buserror_mon.port_ref());
	thisclass::set_local_memory_area(data_type(0), data_type(0));
	mem_ctl.connect_memory(local_mem);
	memory_map().insert(local_mem);
	memory_map().insert(multi_io.memory());
	thisclass::set_processor_number(1);
	thisclass::set_processor_id(0);
}

template <class P, class A, class D, size_t Dsz, class Ch>
generic_processing_element<P, A, D, Dsz, Ch>::generic_processing_element
	(const generic_processing_element<P, A, D, Dsz, Ch>& a)
	: pu(a.pu),
	  mem_ctl(a.mem_ctl),
	  multi_io(a.multi_io),
	  buserror_mon(a.buserror_mon),
	  local_mem(a.local_mem)
{
	insert_synchronous_unit(pu);
	insert_synchronous_unit(mem_ctl);
	insert_synchronous_unit(multi_io);
	insert_synchronous_unit(buserror_mon);
	pu.port_ref().connect(mem_ctl.port_ref());
	pu.port_ref().connect(multi_io.port_ref());
	pu.port_ref().connect(buserror_mon.port_ref());
	thisclass::set_local_memory_area(a.local_mem.top(), a.local_mem.size());
	mem_ctl.connect_memory(local_mem);
	memory_map().insert(local_mem);
	memory_map().insert(multi_io.memory());
	thisclass::set_processor_number(a.processor_number());
	thisclass::set_processor_id(a.processor_id());
}

template <class P, class A, class D, size_t Dsz, class Ch>
generic_processing_element<P, A, D, Dsz, Ch>::~generic_processing_element()
{}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_processor_number
	(size_t a)
{
	multi_io.set_sysinfo("processor_number", data_type(a));
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_processor_id(size_t a)
{
	multi_io.set_sysinfo("processor_id", data_type(a));
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_file_table_size
	(size_t a)
{
	multi_io.set_file_table_size(a);
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_standard_input_stream
	(istream& is)
{
	multi_io.set_file_input_stream(0, is);
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_standard_output_stream
	(ostream& os)
{
	multi_io.set_file_output_stream(1, os);
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_standard_error_stream
	(ostream& os)
{
	multi_io.set_file_output_stream(2, os);
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_commandline_argument
	(const char* const* argv)
{
	multi_io.set_commandline_argument(argv);
}

template <class P, class A, class D, size_t Dsz, class Ch>
int generic_processing_element<P, A, D, Dsz, Ch>::commandline_status(void)
	const
{
	return multi_io.commandline_status();
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_sysinfo
	(const string& a, data_type b)
{
	multi_io.set_sysinfo(a, b);
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_sysinfo
	(const string& a, const string& b)
{
	multi_io.set_sysinfo(a, b);
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::unset_sysinfo
	(const string& a)
{
	multi_io.unset_sysinfo(a);
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_local_memory_area
	(address_type a, size_type b)
{
	local_mem.set_top(a);
	local_mem.resize(b);
	multi_io.set_sysinfo("local_memory_address", a);
	multi_io.set_sysinfo("local_memory_size", b);
	memory_map().update();
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_local_memory_read_wait
	(unsigned int a)
{
	mem_ctl.set_read_wait(a);
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_local_memory_write_wait
	(unsigned int a)
{
	mem_ctl.set_write_wait(a);
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::enable_split_transaction
	(void)
{
	mem_ctl.enable_split_transaction();
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::disable_split_transaction
	(void)
{
	mem_ctl.disable_split_transaction();
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_multi_io_area
	(generic_processing_element<P, A, D, Dsz, Ch>::address_type a,
	 generic_processing_element<P, A, D, Dsz, Ch>::size_type b)
{
	multi_io.set_address(a, b);
	memory_map().update();
}

template <class P, class A, class D, size_t Dsz, class Ch>
void generic_processing_element<P, A, D, Dsz, Ch>::set_bus_error_timeout
	(unsigned int a)
{
	buserror_mon.set_timeout(a);
}

#endif /* GENERIC_PROCESSING_ELEMENT_H */
