/*
 * <<< argument_parser.cc >>>
 *
 * --- Command-line Argument parser class 'argument_parser'
 *     Copyright (C) 1997-1999 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstring>
#include <iostream>
#include "argument_parser.h"

argument_parser::argument_parser(void)
	: short_index(NULL),
	  long_index(NULL),
	  args(NULL)
{}

argument_parser::argument_parser(const char* const* argv)
	: short_index(NULL),
	  long_index(NULL),
	  args(NULL)
{
	set(argv);
}

argument_parser::argument_parser(const argument_parser& a)
{
	assign(a);
}

argument_parser::~argument_parser()
{
	clear();
}

argument_parser& argument_parser::operator=(const argument_parser& a)
{
	if (&a != this) {
		clear();
		assign(a);
	}
	return *this;
}

void argument_parser::set(const char* const* arg)
{
	if (arg == NULL) return;
	// count number of option, number of argument
	int short_number = 0, long_number = 0, args_number = 0;
	const char* const* p = arg;
	while (*p != NULL) {
		if (**p == '-') {
			// option
			if (*(*p + 1) != '-') {
				// "-<char><string>" type option
				if (*(*p + 1) != '\0') short_number++;
				p++;
			} else {
				// "--<string>" type option
				if (*(*p + 2) == '\0') {
					// "--"
					p++;
					break;
				} else {
					// "--<string>"
					long_number++;
					p++;
				}
			}
		} else {
			// not option
			args_number++;
			p++;
		}
	}
	while (*p != NULL) {
		// not option
		args_number++;
		p++;
	}
	// allocate tables
	if (short_number > 0) {
		short_index = new char[short_number + 1];
		short_value = new char*[short_number];
		short_index[short_number] = '\0';
	}
	if (long_number > 0) {
		long_index = new char*[long_number + 1];
		long_value = new char*[long_number];
		long_index[long_number] = NULL;
	}
	args = new char*[args_number + 1];
	args[args_number] = NULL;
	// set option table, argument table
	int short_count = 0, long_count = 0, args_count = 0;
	p = arg;
	while (*p != NULL) {
		if (**p == '-') {
			// option
			if (*(*p + 1) != '-') {
				// "-<char><string>" type option
				if (*(*p + 1) != '\0') {
					int len = strlen(*p + 2);
					short_index[short_count] = *(*p + 1);
					if (len > 0) {
						short_value[short_count] = new char[len + 1];
						strncpy(short_value[short_count], *p + 2, len + 1);
					} else {
						short_value[short_count] = NULL;
					}
					short_count++;
				}
				p++;
			} else {
				// "--<string>" type option
				if (*(*p + 2) == '\0') {
					// "--"
					p++;
					break;
				} else {
					// "--<string>"
					const char* s = *p + 2;
					int len = strlen(s);
					int sep_index = 0;
					while (1) {
						if (s[sep_index] == '=' || s[sep_index] == '\0') {
							break;
						}
						sep_index++;
					}
					long_index[long_count] = new char[sep_index + 1];
					strncpy(long_index[long_count], s, sep_index);
					long_index[long_count][sep_index] = '\0';
					if (sep_index + 1 < len) {
						// "--<string>=<string>"
						long_value[long_count] = new char[len - sep_index];
						strncpy(long_value[long_count], s + sep_index + 1,
								len - sep_index);
					} else {
						// "--<string>=", "--<string>"
						long_value[long_count] = NULL;
					}
					long_count++;
					p++;
				}
			}
		} else {
			// not option
			int len = strlen(*p);
			args[args_count] = new char[len + 1];
			strncpy(args[args_count], *p, len + 1);
			args_count++;
			p++;
		}
	}
	while (*p != NULL) {
		// not option
		int len = strlen(*p);
		args[args_count] = new char[len + 1];
		strncpy(args[args_count], *p, len + 1);
		args_count++;
		p++;
	}
}

void argument_parser::assign(const argument_parser& a)
{
	if (a.short_value != NULL) {
		int number;
		number = strlen(a.short_index);
		short_index = new char[number + 1];
		short_value = new char*[number];
		strncpy(short_index, a.short_index, number + 1);
		for (int i = 0; i < number; i++) {
			if (a.short_value[i] != NULL) {
				int len = strlen(a.short_value[i]);
				short_value[i] = new char[len + 1];
				strncpy(short_value[i], a.short_value[i], len + 1);
			} else {
				short_value[i] = NULL;
			}
		}
	} else {
		short_value = NULL;
	}
	if (a.long_value != NULL) {
		int number;
		for (number = 0; a.long_value[number] != NULL; number++);
		long_value = new char*[number + 1];
		long_index = new char*[number];
		long_value[number] = NULL;
		for (int i = 0; i < number; i++) {
			int len = strlen(long_index[i]);
			long_index[i] = new char[len + 1];
			strncpy(long_index[i], a.long_index[i], len + 1);
			if (a.long_value[i] != NULL) {
				len = strlen(a.long_value[i]);
				long_value[i] = new char[len + 1];
				strncpy(long_value[i], a.long_value[i], len + 1);
			} else {
				long_value[i] = NULL;
			}
		}
	} else {
		long_value = NULL;
	}
	if (a.args != NULL) {
		int number;
		for (number = 0; a.args[number] != NULL; number++);
		args = new char*[number + 1];
		args[number] = NULL;
		for (int i = 0; i < number; i++) {
			int len = strlen(a.args[i]);
			args[i] = new char[len + 1];
			strncpy(args[i], a.args[i], len + 1);
		}
	} else {
		args = NULL;
	}
}

void argument_parser::clear(void)
{
	if (short_index != NULL) {
		for (int i = 0; short_index[i] != '\0'; i++) {
			delete[] short_value[i];
		}
		delete[] short_index;
		delete[] short_value;
		short_value = NULL;
	}
	if (long_index != NULL) {
		for (int i = 0; long_index[i] != NULL; i++) {
			delete[] long_index[i];
			delete[] long_value[i];
		}
		delete[] long_index;
		delete[] long_value;
		long_value = NULL;
	}
	if (args != NULL) {
		for (const char* const* p = args; *p != NULL; p++) {
			delete[] *p;
		}
		delete[] args;
		args = NULL;
	}
}

bool argument_parser::defined(char a) const
{
	if (short_index == NULL) return false;
	for (int i = 0; short_index[i] != '\0'; i++) {
		if (short_index[i] == a) return true;
	}
	return false;
}

bool argument_parser::defined(const char* a) const
{
	if (long_index == NULL) return false;
	for (int i = 0; long_index[i] != NULL; i++) {
		if (strcmp(long_index[i], a) == 0) return true;
	}
	return false;
}

const char* argument_parser::operator[](char a) const
{
	if (short_index == NULL) return NULL;
	for (int i = 0; short_index[i] != '\0'; i++) {
		if (short_index[i] == a) return short_value[i];
	}
	return NULL;
}

const char* argument_parser::operator[](const char* a) const
{
	if (long_index == NULL) return NULL;
	for (int i = 0; long_index[i] != NULL; i++) {
		if (strcmp(long_index[i], a) == 0) return long_value[i];
	}
	return NULL;
}

void argument_parser::output(ostream& os) const
{
	if (short_index == NULL && long_index == NULL && args == NULL) return;
	int short_number = 0, long_number = 0, arg_number = 0;
	if (short_index != NULL) {
		int i;
		for (i = 0; short_index[i] != '\0'; i++);
		short_number = i;
	}
	if (long_index != NULL) {
		int i;
		for (i = 0; long_index[i] != NULL; i++);
		long_number = i;
	}
	if (args != NULL) {
		int i;
		for (i = 0; args[i] != NULL; i++);
		arg_number = i;
	}
	if (short_index != NULL) {
		for (int i = 0; i < short_number; i++) {
			os << '-' << short_index[i];
			if (short_value[i] != NULL) os << short_value[i];
			os << ' ';
		}
	}
	if (long_index != NULL) {
		for (int i = 0; i < long_number; i++) {
			os << "--" << long_index[i];
			if (long_value[i] != NULL) os << '=' << long_value[i];
			os << ' ';
		}
	}
	os << "--";
	if (args != NULL) {
		for (int i = 0; i < arg_number; i++) {
			os << ' ' << args[i];
		}
	}
}

ostream& operator<<(ostream& os, const argument_parser& a)
{
	if (os) a.output(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
	return os;
}
