/*
 * <<< r3010_inst.cc >>>
 *
 * --- Copyright (C) 1996-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstring>
#include <iomanip>
#include <iostream>
#include <strstream>
#include "r3010_inst.h"

#define OP_MASK			0xfc000000
#define OP_SHIFT		26
#define BASE_MASK		0x03e00000
#define BASE_SHIFT		21
#define OP25_MASK		0x02000000
#define OP25_SHIFT		25
#define SUB_MASK		0x01e00000
#define SUB_SHIFT		21
#define BR_MASK			0x001f0000
#define BR_SHIFT		16
#define FMT_MASK		0x01e00000
#define FMT_SHIFT		21
#define RT_MASK			0x001f0000
#define RT_SHIFT		16
#define RS_MASK			0x0000f800
#define RS_SHIFT		11
#define RD_MASK			0x000007c0
#define RD_SHIFT		6
#define FUNC_MASK		0x0000003f
#define FUNC_SHIFT		0
#define OFS_MASK		0x0000ffff
#define OFS_SHIFT		0

#define NONE		0
#define USE_RT		1
#define USE_RS		2
#define USE_RD		4

static int func_reg[] = {
	// ADD.fmt			  SUB.fmt
	USE_RT|USE_RS|USE_RD, USE_RT|USE_RS|USE_RD,
	// MUL.fmt			  DIV.fmt
	USE_RT|USE_RS|USE_RD, USE_RT|USE_RS|USE_RD,
	//					  ABS.fmt
	NONE,				  USE_RS|USE_RD,
	// MOV.fmt			  NEG.fmt
	USE_RS|USE_RD,		  USE_RS|USE_RD,
	// 010
	NONE, NONE, NONE, NONE, NONE, NONE, NONE, NONE,
	// 020
	NONE, NONE, NONE, NONE, NONE, NONE, NONE, NONE,
	// 030
	NONE, NONE, NONE, NONE, NONE, NONE, NONE, NONE,
	// CVT.S.fmt		  CVT.D.fmt
	USE_RS|USE_RD,		  USE_RS|USE_RD,
	//
	NONE, NONE,
	// CVT.W.fmt
	USE_RS|USE_RD,		  NONE,
	//
	NONE, NONE,
	// 050
	NONE, NONE, NONE, NONE, NONE, NONE, NONE, NONE,
	// C.F.fmt	   C.UN.fmt		  C.EQ.fmt		 C.UEQ.fmt
	USE_RT|USE_RS, USE_RT|USE_RS, USE_RT|USE_RS, USE_RT|USE_RS,
	// C.OLT.fmt   C.ULT.fmt	  C.OLE.fmt		 C.ULE.fmt
	USE_RT|USE_RS, USE_RT|USE_RS, USE_RT|USE_RS, USE_RT|USE_RS,
	// C.SF.fmt	   C.NGLE.fmt	  C.SEQ.fmt		 C.NGL.fmt
	USE_RT|USE_RS, USE_RT|USE_RS, USE_RT|USE_RS, USE_RT|USE_RS,
	// C.LT.fmt	   C.NGE.fmt	  C.LE.fmt		 C.NGT.fmt
	USE_RT|USE_RS, USE_RT|USE_RS, USE_RT|USE_RS, USE_RT|USE_RS,
};

void
r3010_inst::reset(void)
{
	_have_inst = false;
	_use_reg[REG_SRC1] = _use_reg[REG_SRC2] = _use_reg[REG_DST] = false;
	_regnum[REG_SRC1] = _regnum[REG_SRC2] = _regnum[REG_DST] = -1;
	_fpa = false;
	_calc = false;
	_bus_from = _bus_to = false;
}

void
r3010_inst::decode(r3000_word w)
{
	_inst = w;
	_have_inst = true;
	_op = (op_t)((w & OP_MASK) >> OP_SHIFT);
	_op25 = (w & OP25_MASK) != 0;
	_sub = (sub_t)((w & SUB_MASK) >> SUB_SHIFT);
	_br = (br_t)((w & BR_MASK) >> BR_SHIFT);
	_fmt = (fmt_t)((w & FMT_MASK) >> FMT_SHIFT);
	_func = (func_t)((w & FUNC_MASK) >> FUNC_SHIFT);
	_base = (w & BASE_MASK) >> BASE_SHIFT;
	_offset = (w & OFS_MASK) >> OFS_SHIFT;
	_reg[ft] = (w & RT_MASK) >> RT_SHIFT;
	_reg[fs] = (w & RS_MASK) >> RS_SHIFT;
	_reg[fd] = (w & RD_MASK) >> RD_SHIFT;
	switch (_op) {
	case OP_COP1:
		if (_op25) {
			if (func_reg[_func]) {
				_fpa = true;
				_calc = true;
				if (func_reg[_func] & USE_RD) {
					_regnum[REG_DST] = _reg[fd];
					_use_reg[REG_DST] = true;
				}
				if (func_reg[_func] & USE_RS) {
					_regnum[REG_SRC1] = _reg[fs];
					_use_reg[REG_SRC1] = true;
				}
				if (func_reg[_func] & USE_RT) {
					_regnum[REG_SRC2] = _reg[ft];
					_use_reg[REG_SRC2] = true;
				}
			} else {
				/*
				  Because of lack of exception handling,
				  rest of instructions are treat as NOP, now,
				  however it has to raise reserved instruction exception.
				*/
			}
		} else {
			// branch & move data from/to R3000
			switch (_sub) {
			case SUB_MF:
				_regnum[REG_SRC1] = _reg[fs];
				_use_reg[REG_SRC1] = true;
				_fpa = true;
				_bus_to = true;
				break;
			case SUB_CF:
				_regnum[REG_SRC1] = _reg[fs] + NFGR;
				_use_reg[REG_SRC1] = true;
				_fpa = true;
				_bus_to = true;
				break;
			case SUB_MT:
				_regnum[REG_DST] = _reg[fs];
				_use_reg[REG_DST] = true;
				_fpa = true;
				_bus_from = true;
				break;
			case SUB_CT:
				_regnum[REG_DST] = _reg[fs] + NFGR;
				_use_reg[REG_DST] = true;
				_fpa = true;
				_bus_from = true;
				break;
			case SUB_BC:
				// R3000 core treats these instructions.
				break;
			default:
				/*
				  Because of lack of exception handling,
				  rest of instructions are treat as NOP, now,
				  however it has to make reserved instruction exception.
				*/
				break;
			}
		}
		break;
	case OP_LWC1:
		_fpa = true;
		_regnum[REG_DST] = _reg[ft];
		_use_reg[REG_DST] = true;
		_bus_from = true;
		break;
	case OP_SWC1:
		_fpa = true;
		_regnum[REG_SRC1] = _reg[ft];
		_use_reg[REG_SRC1] = true;
		_bus_to = true;
		break;
	default:
		// all of rest are not FPA's instructions
		break;
	}
}

/* ------------------------------------------------------------------- */
static const char* func_name[] = {
	"ADD", "SUB", "MUL", "DIV", "004", "ABS", "MOV", "NEG",
	"010", "011", "012", "013", "014", "015", "016", "017",
	"020", "021", "022", "023", "024", "025", "026", "027",
	"030", "031", "032", "033", "034", "035", "036", "037",
	"CVT.S", "CVT.D", "042", "043", "CVT.W", "045", "046", "047",
	"050", "051", "052", "053", "054", "055", "056", "057",
	"C.F",  "C.UN",   "C.EQ",  "C.UEQ", "C.OLT", "C.ULT", "C.OLE", "C.ULE",
	"C.SF", "C.NGLE", "C.SEQ", "C.NGL", "C.LT",  "C.NGE", "C.LE",  "C.NGT",
};

static const char* fmt_name[] = {
	"S", "D", "2", "3", "W", "5", "6", "7",
};

const char*
r3010_inst::disasm() const
{
	ostrstream os;
	if (!have_inst()) {
		os << "*NO INSTRUCTION*";
	} else {
		switch (_op) {
		case OP_COP1:
			disasm_cop1(os);
			break;
		case OP_LWC1:
			os << "LWC1 $f" << _reg[ft] << ", 0x" << hex << _offset << dec
			   << "($r" << _base << ")";
			break;
		case OP_SWC1:
			os << "SWC1 $f" << _reg[ft] << ", 0x" << hex << _offset << dec
			   << "($r" << _base << ")";
			break;
		default:
			os << "[0x" << hex << _inst << dec << "]";
			break;
		}
	}
	os << ends;
	return os.str();
}

void
r3010_inst::disasm_cop1(ostream& os) const
{
	const long flags = os.flags();
	bool f;
	if (_op25) {
		os << func_name[_func] << "." << fmt_name[_fmt] << " ";
		f = false;
		if (_use_reg[REG_DST]) {
			f = true;
			os << "$f" << _regnum[REG_DST];
		}
		if (_use_reg[REG_SRC1]) {
			if( f ) os << ", ";
			f = true;
			os << "$f" << _regnum[REG_SRC1];
		}
		if (_use_reg[REG_SRC2]) {
			if( f ) os << ", ";
			f = true;
			os << "$f" << _regnum[REG_SRC2];
		}
	} else {
		switch (_sub) {
		case SUB_MF:
			os << "MFC1 $r" << _reg[ft] << ", $f" << _reg[fs];
			break;
		case SUB_CF:
			os << "CFC1 $r" << _reg[ft] << ", $f" << _reg[fs];
			break;
		case SUB_MT:
			os << "MTC1 $r" << _reg[ft] << ", $f" << _reg[fs];
			break;
		case SUB_CT:
			os << "CTC1 $r" << _reg[ft] << ", $f" << _reg[fs];
			break;
		case SUB_BC:
			switch (_br) {
			case BR_BC1F:
				os << "BC1F 0x" << hex << _base << dec;
				break;
			case BR_BC1T:
				os << "BC1T 0x" << hex << _base << dec;
				break;
			default:
				os << "???? 0x" << hex << _base << dec;
				break;
			}
			break;
		default:
			os << "???";
			break;
		}
	}
	os.flags(flags);
}

ostream& operator<<(ostream& os, const r3010_inst& inst)
{
	const char *p = inst.disasm();
	os << p;
	delete p;
	return os;
}
