/*
 * <<< r3000_simple_test.cc >>>
 *
 * --- Simple test program for r3000
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <iostream>
#include <isis/memory_control_unit.h>
#include <isis/multi_io_unit.h>
#include <isis/bus_error_detector.h>
#include <isis/r3010.h>
#include "mips_executable.h"
#include "r3000.h"
#include "r3000_directmap_cache.h"
#include <isis/isis.h>

int main(int, char** argv)
{
	typedef r3000_word word;
	const word local_memory_address = 0;
	const word local_memory_size = 0x20000000;
	const word multi_io_area_address = 0xbfd00000;
	const word multi_io_area_size = 0x100;

	// read commandline argument for simulator
	if (*argv == NULL) {
		cerr << "No executable file specified." << endl;
		return 1;
	}

	// create units and connect each other
	r3000 pu;
	memory_control_unit<word> mem_ctl;
	mapped_memory<word> local_mem;
	multi_io_unit<word> multi_io;
	bus_error_detector<word> buserror_mon;
	pu.port_ref().connect(mem_ctl.port_ref());
	pu.port_ref().connect(multi_io.port_ref());
	pu.port_ref().connect(buserror_mon.port_ref());
	mem_ctl.connect_memory(local_mem);

	// setup memory map
	local_mem.set_top(local_memory_address);
	local_mem.resize(local_memory_size);
	multi_io.set_address(multi_io_area_address, multi_io_area_size);
	multi_io.set_sysinfo("local_memory_address", local_memory_address);
	multi_io.set_sysinfo("local_memory_size", local_memory_size);

	// setup file table
	multi_io.set_file_table_size(16);
	multi_io.set_file_input_stream(0, cin);
	multi_io.set_file_output_stream(1, cout);
	multi_io.set_file_output_stream(2, cerr);

	// load program to memory
	{
		ifstream file(argv[1], ios::in | ios::nocreate);
		if (file.bad()) return 1;
		mips_executable obj;
		obj.load_header(file);
		if (!obj.is_valid()) return 1;
		if (!obj.load_body_to_memory(file, local_mem)) return 1;
		pu.program_counter() = obj.entry_point();
	}
	multi_io.set_commandline_argument((const char* const*)(argv + 1));

	// execute simulation
	while (!multi_io.is_halt() && !buserror_mon.is_bus_error()) {
		pu.clock_in();
		mem_ctl.clock_in();
		multi_io.clock_in();
		buserror_mon.clock_in();
		pu.clock_out();
		mem_ctl.clock_out();
		multi_io.clock_out();
		buserror_mon.clock_out();
	}
	return 0;
}
