/*
 * <<< r3000_instruction.cc >>>
 *
 * --- R3000 instruction class 'r3000_instruction'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstring> // needed only for SGI C++ compiler
#include <string>
#include <iomanip> // needed only for SGI C++ compiler
#include <iostream>
#include <strstream>
#include "r3000_instruction.h"
#include "r3000_register_file.h"

template <class T>
static string to_string(T x)
{
	strstream buf;
	T tmp;
	if (x > 0 || x == 0) { // erase warning
		tmp = x;
	} else {
		buf << '-';
		tmp = -x;
	}
	if (tmp == 0 || ((tmp & 0xff) != 0 && ((tmp % 1000) == 0 || tmp < 1000))) {
		buf << tmp;
	} else {
		buf << hex << "0x" << tmp;
	}
	string str;
	buf >> str;
	return str;
}

r3000_instruction::r3000_instruction(void)
{
	clear();
}

r3000_instruction::r3000_instruction(const r3000_instruction& a)
	: image_(a.image_),
	  pc_(a.pc_),
	  opcode_(a.opcode_),
	  operation_(a.operation_)
{}

r3000_instruction::~r3000_instruction()
{}

string r3000_instruction::opcode_string(void) const
{
	static const char* name[] = {
		"lb",		"lbu",		"lh",		"lhu",
		"lw",		"lwl",		"lwr",
		"sb",		"sh",		"sw",		"swl",		"swr",
		"addi",		"addiu",	"slti",		"sltiu",
		"andi",		"ori",		"xori",		"lui",
		"add",		"addu",		"sub",		"subu",		"slt",		"sltu",
		"and",		"or",		"xor",		"nor",
		"sll",		"srl",		"sra",		"sllv",		"srlv",		"srav",
		"mult",		"multu",	"div",		"divu",
		"mfhi",		"mflo",		"mthi",		"mtlo",
		"j",		"jal",		"jr",		"jalr",
		"beq",		"bne",		"blez",		"bgtz",
		"bltz",		"bgez",		"bltzal",	"bgezal",
		"syscall",	"break",
		"lwc",		"swc",		"mtc",		"mfc",		"ctc",		"cfc",
		"cop",		"bc",		"bc",
		"tlbr",		"tlbwi",	"tlbwr",	"tlbp",		"rfe",
		"(\?\?\?)",	"(\?\?\?)"
	};
	string buf(name[opcode()]);
	switch (opcode()) {
	case LWC: case SWC: case MTC: case MFC: case CTC: case CFC:
	case COP: case BCT: case BCF:
		buf += char(coprocessor_number() + '0');
		switch (opcode()) {
		case BCF:
			buf += 'f';
			break;
		case BCT:
			buf += 't';
			break;
		default:
			break;
		}
		break;
	default:
		break;
	}
	return buf;
}

string r3000_instruction::operation_string(void) const
{
	strstream buf;
	switch (opcode()) {
	case LB:  case LBU: case LH: case LHU: case LW:  case LWL: case LWR:
	case SB:  case SH:  case SW: case SWL: case SWR:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rt()) << ", "
				<< to_string(signed_immediate())
				<< '(' << r3000_register_file::register_name(base()) << ')';
		}
		break;
	case LWC: case SWC:
		{
			buf << opcode_string()
				<< " $f" << rt() << ", "
				<< to_string(signed_immediate())
				<< '(' << r3000_register_file::register_name(base()) << ')';
		}
		break;
	case ADDIU:
		if (rs() == 0) {
			buf << "li "
				<< r3000_register_file::register_name(rt()) << ", "
				<< to_string(signed_data_type(signed_immediate()));
		} else {
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rt()) << ", "
				<< r3000_register_file::register_name(rs()) << ", "
				<< to_string(signed_data_type(signed_immediate()));
		}
		break;
	case ADDI: case SLTI: case SLTIU:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rt()) << ", "
				<< r3000_register_file::register_name(rs()) << ", "
				<< to_string(signed_immediate());
		}
		break;
	case ANDI: case ORI: case XORI:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rt()) << ", "
				<< r3000_register_file::register_name(rs()) << ", "
				<< hex << "0x" << immediate();
		}
		break;
	case LUI:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rt()) << ", "
				<< hex << "0x" << immediate();
		}
		break;
	case ADDU:
		if (rs() == 0) {
			buf << "move "
				<< r3000_register_file::register_name(rd()) << ", "
				<< r3000_register_file::register_name(rt());
		} else if (rt() == 0) {
			buf << "move "
				<< r3000_register_file::register_name(rd()) << ", "
				<< r3000_register_file::register_name(rs());
		} else {
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rd()) << ", "
				<< r3000_register_file::register_name(rs()) << ", "
				<< r3000_register_file::register_name(rt());
		}
		break;
	case ADD: case SUB: case SUBU: case SLT: case SLTU:
	case AND: case OR:  case XOR:  case NOR:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rd()) << ", "
				<< r3000_register_file::register_name(rs()) << ", "
				<< r3000_register_file::register_name(rt());
		}
		break;
	case SLL:
		if (image() == 0) {
			buf << "nop";
		} else {
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rd()) << ", "
				<< r3000_register_file::register_name(rt()) << ", "
				<< sa();
		}
		break;
	case SRL: case SRA:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rd()) << ", "
				<< r3000_register_file::register_name(rt()) << ", "
				<< sa();
		}
		break;
	case SLLV: case SRLV: case SRAV:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rd()) << ", "
				<< r3000_register_file::register_name(rt()) << ", "
				<< r3000_register_file::register_name(rs());
		}
		break;
	case MULT: case MULTU: case DIV: case DIVU:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rs()) << ", "
				<< r3000_register_file::register_name(rt());
		}
		break;
	case MFHI: case MFLO:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rd());
		}
		break;
	case MTHI: case MTLO: case JR:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rs());
		}
		break;
	case J: case JAL:
		{
			buf << opcode_string() << ' '
				<< hex << "0x" << jump_address();
		}
		break;
	case JALR:
		{
			buf << opcode_string() << ' ';
			if (rd() != 31) {
				buf << r3000_register_file::register_name(rd()) << ", ";
			}
			buf << r3000_register_file::register_name(rs());
		}
		break;
	case BEQ:
		{
			if (rs() == rt()) {
				buf << "b ";
			} else {
				buf << opcode_string() << ' '
					<< r3000_register_file::register_name(rs()) << ", "
					<< r3000_register_file::register_name(rt()) << ", ";
			}
			buf << "0x" << hex << branch_address();
		}
		break;
	case BNE:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rs()) << ", "
				<< r3000_register_file::register_name(rt()) << ", "
				<< "0x" << hex << branch_address();
		}
		break;
	case BLEZ: case BGTZ: case BLTZ: case BGEZ: case BLTZAL: case BGEZAL:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rs()) << ", "
				<< "0x" << hex << branch_address();
		}
		break;
	case BREAK:
		{
			buf << opcode_string() << ' ' << to_string(break_code());
		}
		break;
	case SYSCALL:
	case TLBR: case TLBWI: case TLBWR: case TLBP: case RFE:
		{
			buf << opcode_string();
		}
		break;
	case MTC: case MFC:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rt())
				<< ", $f" << rd();
		}
		break;
	case CTC: case CFC:
		{
			buf << opcode_string() << ' '
				<< r3000_register_file::register_name(rt())
				<< ", $" << rd();
		}
		break;
	case BCT: case BCF:
		{
			buf << opcode_string() << ' '
				<< "0x" << hex << branch_address();
		}
		break;
	case COP: // coprocessor
		buf << "-- cop --";
		break;
	default:
		buf << opcode_string();
		break;
	}
	return string(buf.str(), buf.pcount());
}

void r3000_instruction::set(address_type a, data_type b)
{
	static const opcode_type opcode_table[] = {
	  SPECIAL_,BCOND_,  J,       JAL,     BEQ,     BNE,     BLEZ,    BGTZ,
	  ADDI,    ADDIU,   SLTI,    SLTIU,   ANDI,    ORI,     XORI,    LUI,
	  COP_,    COP_,    COP_,    COP_,    RESERVED,RESERVED,RESERVED,RESERVED,
	  RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,
	  LB,      LH,      LWL,     LW,      LBU,     LHU,     LWR,     RESERVED,
	  SB,      SH,      SWL,     SW,      RESERVED,RESERVED,SWR,     RESERVED,
	  LWC,     LWC,     LWC,     LWC,     RESERVED,RESERVED,RESERVED,RESERVED,
	  SWC,     SWC,     SWC,     SWC,     RESERVED,RESERVED,RESERVED,RESERVED
	};
	static const opcode_type special_table[] = {
	  SLL,     RESERVED,SRL,     SRA,     SLLV,    RESERVED,SRLV,    SRAV,    
	  JR,      JALR,    RESERVED,RESERVED,SYSCALL, BREAK,   RESERVED,RESERVED,
	  MFHI,    MTHI,    MFLO,    MTLO,    RESERVED,RESERVED,RESERVED,RESERVED,
	  MULT,    MULTU,   DIV,     DIVU,    RESERVED,RESERVED,RESERVED,RESERVED,
	  ADD,     ADDU,    SUB,     SUBU,    AND,     OR,      XOR,     NOR,     
	  RESERVED,RESERVED,SLT,     SLTU,    RESERVED,RESERVED,RESERVED,RESERVED,
	  RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,
	  RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,RESERVED,RESERVED
	};
	static const opcode_type bcond_table[] = {
	  BLTZ,    BGEZ,    UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN,
	  UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN,
	  BLTZAL,  BGEZAL,  UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN,
	  UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN
	};
	static const opcode_type cop_table[] = {
	  MFC,     UNKNOWN, CFC,     UNKNOWN, MTC,     UNKNOWN, CTC,     UNKNOWN,
	  BC_,     UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN,
	  COP,     COP,     COP,     COP,     COP,     COP,     COP,     COP,
	  COP,     COP,     COP,     COP,     COP,     COP,     COP,     COP
	};
	static const opcode_type cop_bc_table[] = {
	  BCF,     BCT,     UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN,
	  UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN,
	  UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN,
	  UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN
	};
	static const opcode_type cop0_table[] = {
	  UNKNOWN, TLBR,    TLBWI,   UNKNOWN, UNKNOWN, UNKNOWN, TLBWR,   UNKNOWN,
	  TLBP,    UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN,
	  RFE,     UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN,
	  UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN, UNKNOWN
	};
	pc_ = a, image_ = b;
	opcode_ = opcode_table[int(image() >> 26)];
	switch (opcode_) {
	case SPECIAL_:
		operation_ = R_TYPE;
		opcode_ = special_table[int(image() & 0x3f)];
		break;
	case BCOND_:
		operation_ = I_TYPE;
		opcode_ = bcond_table[int((image() >> 16) & 0x1f)];
		break;
	case COP_:
		operation_ = CO_TYPE;
		opcode_ = cop_table[int((image() >> 21) & 0x1f)];
		if (opcode_ == COP && coprocessor_number() == 0) {
			opcode_ = cop0_table[int(image() & 0x1f)];
		} else if (opcode_ == BC_) {
			opcode_ = cop_bc_table[int((image() >> 16) & 0x1f)];
		}
		break;
	case J: case JAL:
		operation_ = J_TYPE;
		break;
	default:
		operation_ = I_TYPE;
		break;
	}
}

void r3000_instruction::output(ostream& os) const
{
	os << operation_string();
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

ostream& operator<<(ostream& os, const r3000_instruction& a)
{
	if (os) a.output(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
	return os;
}
