/*
 * <<< r3000.cc >>>
 *
 * --- R3000 class 'r3000'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <string>  // needed only for SGI C++ compiler
#include <iomanip> // needed only for SGI C++ compiler
#include <iostream>
#include <isis/r3010.h>
#include "r3000.h"
#include "r3000_directmap_cache.h"
#include "r3000_bus_interface.h"
#include "r3000_memory_access_unit.h"
#include "r3000_write_buffer_unit.h"
#include "r3000_integer_unit.h"
#include "r3000_cp0.h"

r3000::r3000(void)
	: bus_if(new r3000_bus_interface),
	  cp0(new r3000_cp0),
	  ma_unit(new r3000_memory_access_unit(*bus_if, cp0->register_file())),
	  wb_unit(new r3000_write_buffer_unit(*ma_unit)),
	  int_unit(new r3000_integer_unit(*ma_unit, *wb_unit, *cp0)),
	  halt_flag_(false),
	  freq_mag_(1)
{}

r3000::r3000(const r3000& a)
	: bus_if(new r3000_bus_interface(*(a.bus_if))),
	  cp0(new r3000_cp0(*(a.cp0))),
	  ma_unit(new r3000_memory_access_unit
	  			(*(a.ma_unit), *bus_if, cp0->register_file())),
	  wb_unit(new r3000_write_buffer_unit(*(a.wb_unit), *ma_unit)),
	  int_unit(new r3000_integer_unit(*(a.int_unit), *ma_unit, *wb_unit, *cp0)),
	  halt_flag_(a.halt_flag_),
	  freq_mag_(a.freq_mag_)
{}

r3000::~r3000()
{
	delete int_unit;
	delete wb_unit;
	delete ma_unit;
	delete cp0;
	delete bus_if;
}

r3000::data_type r3000::register_file(size_t i) const
{
	return int_unit->register_file(i);
}

r3000::data_type& r3000::register_file(size_t i)
{
	return int_unit->register_file(i);
}

r3000::address_type r3000::program_counter(void) const
{
	return int_unit->pc();
}

r3000::address_type& r3000::program_counter(void)
{
	return int_unit->pc();
}

bool r3000::is_halt(void) const
{
	return halt_flag_;
}

bool r3000::is_stall(void) const
{
	return int_unit->is_stall_cycle();
}

bool r3000::is_reading(void) const
{
	return ma_unit->is_reading();
}

bool r3000::is_writing(void) const
{
	return ma_unit->is_writing();
}

bool r3000::is_user_mode(void) const
{
	return cp0->is_user_mode();
}

bool r3000::is_kernel_mode(void) const
{
	return cp0->is_kernel_mode();
}

r3000::data_type r3000::hi(void) const
{
	return int_unit->hi();
}

r3000::data_type r3000::lo(void) const
{
	return int_unit->lo();
}

void r3000::clock_in(void)
{
	ma_unit->clock_in();
	wb_unit->clock_in();
	if (freq_mag_ > 1 && !int_unit->is_external_cause_stall_cycle()) {
		for (int i = 1; i < freq_mag_; i++) {
			int_unit->clock();
			if (int_unit->is_external_cause_stall_cycle()) break;
		}
	}
	int_unit->clock();
}

void r3000::clock_out(void)
{
	wb_unit->clock_out();
	ma_unit->clock_out();
}

void r3000::reset(void)
{
	ma_unit->reset();
	wb_unit->reset();
	int_unit->reset();
	cp0->reset();
}

void r3000::halt(void)
{
	halt_flag_ = true;
}

size_t r3000::frequency_magnification(void) const
{
	return freq_mag_;
}

void r3000::set_frequency_magnification(size_t a)
{
	if (a == 1) {
		enable_streaming();
	} else {
		disable_streaming();
	}
	freq_mag_ = a;
}

const port& r3000::port_ref(void) const
{
	return *bus_if;
}

port& r3000::port_ref(void)
{
	return *bus_if;
}

bool r3000::is_connected_to_instruction_cache(void) const
{
	return ma_unit->is_connected_to_instruction_cache();
}

bool r3000::is_connected_to_data_cache(void) const
{
	return ma_unit->is_connected_to_data_cache();
}

bool r3000::is_connected_to_coprocessor(void) const
{
	return int_unit->is_connected_to_coprocessor();
}

void r3000::connect_instruction_cache(r3000_directmap_cache& a)
{
	if (a.cache_size() > 0) ma_unit->connect_instruction_cache(a);
}

void r3000::connect_data_cache(r3000_directmap_cache& a)
{
	if (a.cache_size() > 0) ma_unit->connect_data_cache(a);
}

void r3000::connect_coprocessor(r3010& a)
{
	int_unit->connect_coprocessor(a);
}

void r3000::disconnect_instruction_cache(void)
{
	ma_unit->disconnect_instruction_cache();
}

void r3000::disconnect_data_cache(void)
{
	ma_unit->disconnect_data_cache();
}

void r3000::disconnect_coprocessor(void)
{
	int_unit->disconnect_coprocessor();
}

void r3000::enable_streaming(void)
{
	int_unit->enable_streaming();
}

void r3000::disable_streaming(void)
{
	int_unit->disable_streaming();
}

void r3000::output(ostream& os) const
{
	os << int_unit;
#ifdef DEBUG
	os << "\nwbu: " << wb_unit
	   << "\nmau: " << ma_unit
	   << "\nbus: " << bus_if;
	os.flush();
#endif // DEBUG
}

bool r3000::output(ostream& os, const string& s) const
{
	if (s == "bus_if" || s == "busif") {
		os << bus_if;
		return true;
	} else if (s == "ma_unit" || s == "mau") {
		os << ma_unit;
		return true;
	} else if (s == "write_buffer" || s == "wbuf") {
		os << wb_unit;
		return true;
	} else if (s == "cp0") {
		os << cp0;
		return true;
	} else if (s == "int_unit" || s == "int") {
		os << int_unit;
		return true;
	}
	return int_unit->output(os, s);
}
