/* -*- C++ -*-
 *
 * <<< virtual_channel.h >>>
 *
 * --- Virtual channel class 'virtual_channel'
 *     Copyright (C) 1997-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef VIRTUAL_CHANNEL_H
#define VIRTUAL_CHANNEL_H 1

#include <iostream>
#include <vector>
#include <isis/cyclic_queue.h>
#include <isis/root_object.h>

class packet;
class virtual_channel_input;
class virtual_channel_output;

class virtual_channel_input : public root_object
{
private:
	typedef virtual_channel_input thisclass;
	typedef root_object inherited;
	typedef virtual_channel_output partner_type;
	partner_type* partner;
	vector< cyclic_queue<packet*> > buf;
public:
	virtual_channel_input(void);
	virtual_channel_input(size_t, size_t);
	virtual_channel_input(const thisclass&);
	virtual ~virtual_channel_input();
	virtual bool check_invariant(void) const;
	virtual void output(ostream&) const;
	bool is_connected(void) const { return partner != NULL; }
	bool is_connected(const partner_type& a) const
		{ return partner == &a; }
	void connect(partner_type&);
	void disconnect(void);
	size_t channel_size(void) const
		{ return buf.size(); }
	size_t buffer_size(void) const
		{ return (buf.size() > 0) ? buf[0].max_size() : 0; }
	void set_channel_size(size_t);
	void set_buffer_size(size_t);
	bool empty(size_t a) const { return buf[a].empty(); }
	bool full(size_t a) const { return buf[a].full(); }
	size_t length(size_t a) const { return buf[a].size(); }
	void put(size_t a, packet* b) { buf[a].push(b); }
	packet* get(size_t a)
		{ packet* t = buf[a].front();  buf[a].pop();  return t; }
	const packet* look(size_t a) const { return buf[a].front(); }
	packet* look(size_t a) { return buf[a].front(); }
	void del(size_t a) { buf[a].pop(); }
	void clear(size_t);
	void clear(void);
};

class virtual_channel_output : public root_object
{
private:
	typedef virtual_channel_output thisclass;
	typedef root_object inherited;
	typedef virtual_channel_input partner_type;
	partner_type* partner;
public:
	virtual_channel_output(void);
	virtual_channel_output(const thisclass&);
	virtual ~virtual_channel_output();
	virtual bool check_invariant(void) const;
	virtual void output(ostream&) const;
	bool is_connected(void) const { return partner != NULL; }
	bool is_connected(const partner_type& a) const
		{ return partner == &a; }
	void connect(partner_type& a) { a.connect(*this); }
	void disconnect(void) { if (is_connected()) partner->disconnect(); }
	size_t channel_size(void) const { return partner->channel_size(); }
	size_t buffer_size(void) const { return partner->buffer_size(); }
	void set_channel_size(size_t a) { partner->set_channel_size(a); }
	void set_buffer_size(size_t a) { partner->set_buffer_size(a); }
	bool empty(size_t a) const { return partner->empty(a); }
	bool full(size_t a) const { return partner->full(a); }
	size_t length(size_t a) const { return partner->length(a); }
	void put(size_t a, packet* b) { partner->put(a, b); }
	const packet* look(size_t a) const { return partner->look(a); }
	packet* look(size_t a) { return partner->look(a); }
	void clear(size_t a) { partner->clear(a); }
	void clear(void) { partner->clear(); }
	friend void virtual_channel_input::connect(virtual_channel_output&);
	friend void virtual_channel_input::disconnect(void);
};

#endif /* VIRTUAL_CHANNEL_H */
