/*
 * <<< network_packet_base.h >>>
 *
 * --- Network packet base class 'network_packet_base'
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef NETWORK_PACKET_BASE_H
#define NETWORK_PACKET_BASE_H

#include <iostream>
#include <vector>
#include <isis/packet.h>

template <class A>
class network_packet_base : public packet
{
private:
	typedef network_packet_base<A> thisclass;
	typedef packet inherited;
public:
	typedef A address_type;
	typedef size_t size_type;
	typedef unsigned long timestamp_type;
private:
	typedef vector<address_type> address_container_type;
	enum destination_set_type { UNICAST, MULTICAST, BROADCAST };
	address_type src_;
	address_type ucast_dst_; // for unicast
	address_container_type mcast_dst_tbl_; // for multicast
	destination_set_type dst_set_;
	size_type len_;
	timestamp_type gen_time_;
public:
	network_packet_base(void) {}
	network_packet_base(const thisclass& a)
		: src_(a.src_), ucast_dst_(a.ucast_dst_),
		  mcast_dst_tbl_(a.mcast_dst_tbl_), dst_set_(a.dst_set_),
		  len_(a.len_), gen_time_(a.gen_time_) {}
	virtual ~network_packet_base() {}
	virtual root_object* new_object(void) const
		{ return new thisclass; }
	virtual root_object* clone_object(void) const
		{ return new thisclass(*this); }
	virtual void output(ostream&) const;
	// functions for source and destination(es) address
	bool is_unicast(void) const { return dst_set_ == UNICAST; }
	bool is_multicast(void) const { return dst_set_ == MULTICAST; }
	bool is_broadcast(void) const { return dst_set_ == BROADCAST; }
	address_type source(void) const { return src_; }
	address_type destination(void) const { return ucast_dst_; }
	size_t multicast_destination_number(void) const
		{ return mcast_dst_tbl_.size(); }
	address_type multicast_destination(size_t i) const
		{ return mcast_dst_tbl_[i]; }
	void set_unicast(void) { dst_set_ = UNICAST; }
	void set_multicast(void) { dst_set_ = MULTICAST; }
	void set_broadcast(void) { dst_set_ = BROADCAST; }
	void set_source(address_type a) { src_ = a; }
	void set_destination(address_type a) { ucast_dst_ = a; }
	void set_multicast_destination_number(size_t n)
		{ mcast_dst_tbl_.resize(n); }
	void set_multicast_destination(size_t i, address_type a)
		{ mcast_dst_tbl_[i] = a; }
	// functions for packet length
	size_type length(void) const { return len_; }
	void set_length(size_type a) { len_ = a; }
	// functions for timestamp
	timestamp_type timestamp(void) const { return gen_time_; }
	void set_timestamp(timestamp_type a) { gen_time_ = a; }
};

template <class A>
void network_packet_base<A>::output(ostream& os) const
{
	os << '(' << source() << '>';
	if (is_unicast()) {
		os << destination();
	} else if (is_multicast()) {
		for (size_t i = 0; i << multicast_destination_number(); i++) {
			os << multicast_destination(i);
			if (i < multicast_destination_number() - 1) os << ',';
		}
	} else if (is_broadcast()) {
		os << '*';
	}
	os << ')';
}

#endif /* NETWORK_PACKET_BASE_H */
