/* -*- C++ -*-
 *
 * <<< memory_map.h >>>
 *
 * --- Memory map class 'memory_map'
 *     Copyright (C) 1997-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef MEMORY_MAP_H
#define MEMORY_MAP_H 1

#include <iostream>
#include <vector>
#include <isis/root_object.h>
#include <isis/mapped_memory.h>

template <class A, class D = A, size_t Dsz = sizeof(D), class Ch = char>
class memory_map : public root_object
{
private:
	typedef memory_map<A, D, Dsz, Ch> thisclass;
	typedef root_object inherited;
	typedef mapped_memory<A, D, Dsz, Ch> mapped_memory_type;
	typedef vector<mapped_memory_type*> container_type;
	container_type buf;
public:
	typedef typename mapped_memory_type::address_type address_type;
	typedef typename mapped_memory_type::data_type data_type;
	typedef typename mapped_memory_type::char_type char_type;
	typedef typename mapped_memory_type::size_type size_type;
	static const size_t sizeof_data_type = mapped_memory_type::sizeof_data_type;
	memory_map(void);
	memory_map(const thisclass&);
	virtual ~memory_map();
	virtual void output(ostream&) const;
	bool is_valid(address_type) const;
	data_type read(address_type) const;
	char_type read_char(address_type) const;
	void write(address_type, data_type);
	void write_char(address_type, char_type);
	void insert(mapped_memory_type&);
	void erase(mapped_memory_type&);
	void update(void);
	void clear(void);
	void dump(ostream&) const;
	void dump(ostream&, address_type, size_type) const;
};

template <class A, class D, size_t Dsz, class Ch>
memory_map<A, D, Dsz, Ch>::memory_map(void)
{}

template <class A, class D, size_t Dsz, class Ch>
memory_map<A, D, Dsz, Ch>::memory_map(const memory_map<A, D, Dsz, Ch>& a)
	: buf(a.buf)
{}

template <class A, class D, size_t Dsz, class Ch>
memory_map<A, D, Dsz, Ch>::~memory_map()
{}

template <class A, class D, size_t Dsz, class Ch>
bool memory_map<A, D, Dsz, Ch>::is_valid
	(memory_map<A, D, Dsz, Ch>::address_type a) const
{
	typedef typename container_type::const_iterator CI;
	for (CI p = buf.begin(); p != buf.end(); p++) {
		if ((*p)->is_valid(a)) return true;
	}
	return false;
}

template <class A, class D, size_t Dsz, class Ch>
memory_map<A, D, Dsz, Ch>::data_type memory_map<A, D, Dsz, Ch>::read
	(memory_map<A, D, Dsz, Ch>::address_type a) const
{
	typedef typename container_type::const_iterator CI;
	for (CI p = buf.begin(); p != buf.end(); p++) {
		if ((*p)->is_valid(a)) return (*p)->read(a);
	}
	return data_type(0);
}

template <class A, class D, size_t Dsz, class Ch>
memory_map<A, D, Dsz, Ch>::char_type memory_map<A, D, Dsz, Ch>::read_char
	(memory_map<A, D, Dsz, Ch>::address_type a) const
{
	typedef typename container_type::const_iterator CI;
	for (CI p = buf.begin(); p != buf.end(); p++) {
		if ((*p)->is_valid(a)) return (*p)->read_char(a);
	}
	return char_type(0);
}

template <class A, class D, size_t Dsz, class Ch>
void memory_map<A, D, Dsz, Ch>::write
	(memory_map<A, D, Dsz, Ch>::address_type a,
	 memory_map<A, D, Dsz, Ch>::data_type b)
{
	typedef typename container_type::iterator IT;
	for (IT p = buf.begin(); p != buf.end(); p++) {
		if ((*p)->is_valid(a)) (*p)->write(a, b);
	}
}

template <class A, class D, size_t Dsz, class Ch>
void memory_map<A, D, Dsz, Ch>::write_char
	(memory_map<A, D, Dsz, Ch>::address_type a,
	 memory_map<A, D, Dsz, Ch>::char_type b)
{
	typedef typename container_type::iterator IT;
	for (IT p = buf.begin(); p != buf.end(); p++) {
		if ((*p)->is_valid(a)) (*p)->write_char(a, b);
	}
}

template <class A, class D, size_t Dsz, class Ch>
void memory_map<A, D, Dsz, Ch>::insert
	(memory_map<A, D, Dsz, Ch>::mapped_memory_type& a)
{
	typename container_type::iterator p;
	for (p = buf.begin(); p != buf.end(); p++) {
		if ((*p)->top() > a.top()) break;
	}
	buf.insert(p, &a);
}

template <class A, class D, size_t Dsz, class Ch>
void memory_map<A, D, Dsz, Ch>::erase
	(memory_map<A, D, Dsz, Ch>::mapped_memory_type& a)
{
	typename container_type::iterator p;
	for (p = buf.begin(); p != buf.end(); p++) {
		if (p == &a) {
			buf.erase(&a);
			break;
		}
	}
}

template <class A, class D, size_t Dsz, class Ch>
void memory_map<A, D, Dsz, Ch>::update(void)
{
	typedef typename container_type::iterator IT;
	container_type tmp(buf);
	buf.clear();
	while (!tmp.empty()) {
		insert(*(tmp.back()));
		tmp.pop_back();
	}
}

template <class A, class D, size_t Dsz, class Ch>
void memory_map<A, D, Dsz, Ch>::clear(void)
{
	buf.clear();
}

template <class A, class D, size_t Dsz, class Ch>
void memory_map<A, D, Dsz, Ch>::dump(ostream& os) const
{
	typedef typename container_type::const_iterator CI;
	for (CI p = buf.begin(); p != buf.end(); p++) {
		if (p != buf.begin()) os << '\n';
		(*p)->dump(os);
	}
}

template <class A, class D, size_t Dsz, class Ch>
void memory_map<A, D, Dsz, Ch>::dump(ostream& os,
	memory_map<A, D, Dsz, Ch>::address_type a,
	memory_map<A, D, Dsz, Ch>::size_type b) const
{
	vector<bool> flags(buf.size());
	int count = 0, i;
	typename container_type::const_iterator p;
	for (p = buf.begin(), i = 0; p != buf.end(); p++, i++) {
		if ((*p)->is_valid(a) || (*p)->is_valid(a + b - sizeof(data_type)) ||
			(a <= (*p)->top() && (a + b) >= (*p)->bottom())) {
			flags[i] = true;
			count++;
		} else {
			flags[i] = false;
		}
	}
	for (i = 0, p = buf.begin(); count > 0; i++, p++) {
		if (flags[i]) {
			(*p)->dump(os, a, b);
			if (--count > 0) os << '\n';
		}
	}
}

template <class A, class D, size_t Dsz, class Ch>
void memory_map<A, D, Dsz, Ch>::output(ostream& os) const
{
	dump(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* MEMORY_MAP_H */
