/* -*- C++ -*-
 *
 * <<< tracemaker_element.h >>>
 *
 * --- Processing Element for Tracemaker class 'tracemaker_element'
 *     Copyright (C) 1997-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef TRACEMAKER_ELEMENT_H
#define TRACEMAKER_ELEMENT_H 1

#include <iostream>
#include "r3081_processing_element.h"
#include "bus_port.h"
#include "mapped_memory.h"
#include "memory_control_unit.h"
#include "fetch_and_dec_control_unit.h"

class tracemaker_element : public r3081_processing_element
{
private:
	typedef tracemaker_element thisclass;
	typedef r3081_processing_element inherited;
public:
	typedef inherited::address_type address_type;
	typedef inherited::data_type data_type;
	typedef inherited::char_type char_type;
	typedef inherited::size_type size_type;
	typedef inherited::processor_type processor_type;
	typedef inherited::memory_map_type memory_map_type;
	static const size_type sizeof_data_type = inherited::sizeof_data_type;
	typedef mapped_memory<address_type, data_type, sizeof_data_type, char_type>
		mapped_memory_type;
private:
	memory_control_unit<address_type, data_type, sizeof_data_type, char_type>
		shmem_ctl;
	fetch_and_dec_control_unit<address_type, data_type, sizeof_data_type,
							   char_type>
		shfad_ctl;
	bus_port<address_type, data_type> bus_if;
	// for trace output
	mapped_memory_type* shmem;
	mapped_memory_type* shfad;
	mutable address_type old_adr;
	bool all_trace_flag;
public:
	tracemaker_element(void);
	tracemaker_element(const thisclass&);
	virtual ~tracemaker_element();
	thisclass& operator=(const thisclass&);
	void connect_shared_memory(mapped_memory_type&);
	void connect_sync_memory(mapped_memory_type&);
	void update_shared_memory_area(void);
	void update_sync_memory_area(void);
	inline void output_trace(ostream&) const;
	void enable_all_trace(void) { all_trace_flag = 1; }
	void disable_all_trace(void) { all_trace_flag = 0; }
	unsigned int shared_memory_read_wait(void) const
		{ return shmem_ctl.read_wait(); }
	unsigned int shared_memory_write_wait(void) const
		{ return shmem_ctl.write_wait(); }
	unsigned int sync_memory_read_wait(void) const
		{ return shfad_ctl.read_wait(); }
	unsigned int sync_memory_write_wait(void) const
		{ return shfad_ctl.write_wait(); }
	void set_shared_memory_read_wait(unsigned int);
	void set_shared_memory_write_wait(unsigned int);
	void set_sync_memory_read_wait(unsigned int);
	void set_sync_memory_write_wait(unsigned int);
};

inline void tracemaker_element::output_trace(ostream& os) const
{
	address_type adr;
	if (bus_if.is_request() && (adr = bus_if.address()) != old_adr &&
		(all_trace_flag || shmem->is_valid(adr) || shfad->is_valid(adr))) {
		const long flags = os.flags();
		os << hex
		   << setw(8) << timer_clock_value()
		   << setw(4) << processor_id()
		   << (bus_if.is_read_request() ? " r " : " w ")
		   << setw(8) << adr << '\n';
		os.flags(flags);
		old_adr = adr;
	}
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* TRACEMAKER_ELEMENT_H */
