/*
 * <<< simple_access_test.cc >>>
 *
 * --- Test program for cache_controller class
 *     Copyright (C) 2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <iostream>
// isis
#include "argument_parser.h"
#include "bus_port.h"
#include "memory_control_unit.h"
#include "mapped_memory.h"
//
#include "cache_protocol.h"
#include "set_associative_cache.h"
#include "cache_controller.h"
#include "big_word.h"
#include "bus_bridge.h"

typedef unsigned int word;
typedef cache_line_PROTOCOL<word, word> cache_line_t;
typedef set_associative_cache<cache_line_t> cache_t;
typedef cache_controller<word> cache_ctl_t;
typedef big_word<word> bus_data_t;

static void single_read_test(cache_ctl_t[], cache_t[], bus_port<word>[],
							 bus_port<word>&, bus_port<bus_data_t>&,
							 mapped_memory<word, word>&,
							 mapped_memory<word, word>[],
							 memory_control_unit<word>&, bus_bridge<word>&);
static void single_write_test(cache_ctl_t[], cache_t[], bus_port<word>[],
							  bus_port<word>&, bus_port<bus_data_t>&,
							  mapped_memory<word, word>&,
							  mapped_memory<word, word>[],
							  memory_control_unit<word>&, bus_bridge<word>&);
static void multi_read_test(cache_ctl_t[], cache_t[], bus_port<word>[],
							bus_port<word>&, bus_port<bus_data_t>&,
							mapped_memory<word, word>&,
							mapped_memory<word, word>[],
							memory_control_unit<word>&, bus_bridge<word>&);
static void multi_write_test(cache_ctl_t[], cache_t[], bus_port<word>[],
							 bus_port<word>&, bus_port<bus_data_t>&,
							 mapped_memory<word, word>&,
							 mapped_memory<word, word>[],
							 memory_control_unit<word>&, bus_bridge<word>&);

const int punum = 2;
bool VerboseMode = true;
int global_clock = 0;

int main(int, char *argv[])
{
	word cache_size = 0x20, line_size = 0x8, way_size = 2;
	word sh_size = 0x20, sy_size = 0x10;
	{
		argument_parser arg_parse((const char* const*)(argv + 1));
		if (arg_parse['c'] != NULL) cache_size = atoi(arg_parse['c']);
		if (arg_parse['f'] != NULL)    sy_size = atoi(arg_parse['f']);
		if (arg_parse['l'] != NULL)  line_size = atoi(arg_parse['l']);
		if (arg_parse['n'] != NULL)    sh_size = atoi(arg_parse['n']);
		if (arg_parse['w'] != NULL)   way_size = atoi(arg_parse['w']);
	}
	const word sh_top = 0, sy_top = sh_size;
	const word top = 0, size = sh_size + sy_size;
	cache_ctl_t cache_ctl[punum];
	cache_t cache_buf[punum];
	bus_port<word> lobus_if[punum], exbus_if;
	bus_port<bus_data_t> shbus_if;
	mapped_memory<word, word> mem_src(top, size), mem_pu[punum];
	memory_control_unit<word> mem_ctl;
	bus_bridge<word> bridge;
	arbiter sender(1, punum);
	for (int i = 0; i < punum; i++) {
		cache_buf[i].resize(cache_size, line_size, way_size);
		mem_pu[i].set_top(top);
		mem_pu[i].resize(size);
		cache_ctl[i].set_shared_address(sh_top, sh_size);
		cache_ctl[i].set_sync_address(sy_top, sy_size);
		cache_ctl[i].connect_cache(cache_buf[i]);
		cache_ctl[i].connect_local_bus_port(lobus_if[i]);
		cache_ctl[i].connect_shared_bus_port(shbus_if);
		cache_ctl[i].connect_sender_arb(sender);
		cache_ctl[i].connect_mem(&mem_src, &mem_src);
		cache_ctl[i].connect_mem_port(exbus_if);
	}
	mem_ctl.connect_memory(mem_src);
	mem_ctl.port_ref().connect(exbus_if);
	bridge.connect_external_bus_port(exbus_if);
	bridge.connect_internal_bus_port(shbus_if);

	// memory setup
	mem_src.clear();
	for (int i = 0; i < punum; i++) mem_pu[i].clear();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		mem_src.write(adr, adr);
	}

	cout << "=== single processor read test" << endl;
	cerr << "=== single processor read test" << endl;
	single_read_test(cache_ctl, cache_buf, lobus_if, exbus_if, shbus_if,
					 mem_src, mem_pu, mem_ctl, bridge);
	cout << "mem:" << endl << mem_src << endl
		 << "pu:" << endl << mem_pu[0] << endl
		 << "cache:" << endl << cache_buf[0] << endl;

	cout << "=== single processor write test" << endl;
	cerr << "=== single processor write test" << endl;
	single_write_test(cache_ctl, cache_buf, lobus_if, exbus_if, shbus_if,
					  mem_src, mem_pu, mem_ctl, bridge);
	cout << "mem:" << endl << mem_src << endl
		 << "pu:" << endl << mem_pu[0] << endl
		 << "cache:" << endl << cache_buf[0] << endl;

	cout << "=== multi processor read test" << endl;
	cerr << "=== multi processor read test" << endl;
	multi_read_test(cache_ctl, cache_buf, lobus_if, exbus_if, shbus_if,
					mem_src, mem_pu, mem_ctl, bridge);
	cout << "mem:" << endl << mem_src << endl;
	for (int i = 0; i < punum; i++) {
		cout << "pu" << i << ':' << endl << mem_pu[i] << endl
			 << "cache" << i << ':' << endl << cache_buf[i] << endl;
	}

	cout << "=== multi processor write test" << endl;
	cerr << "=== multi processor write test" << endl;
	multi_write_test(cache_ctl, cache_buf, lobus_if, exbus_if, shbus_if,
					 mem_src, mem_pu, mem_ctl, bridge);
	cout << "mem:" << endl << mem_src << endl;
	for (int i = 0; i < punum; i++) {
		cout << "pu" << i << ':' << endl << mem_pu[i] << endl
			 << "cache" << i << ':' << endl << cache_buf[i] << endl;
	}

	return 0;
}

void single_read_test(cache_ctl_t cache_ctl[],
					  cache_t cache_buf[],
					  bus_port<word> lobus_if[],
					  bus_port<word>& exbus_if,
					  bus_port<bus_data_t>& shbus_if,
					  mapped_memory<word, word>& mem_src,
					  mapped_memory<word, word> mem_pu[],
					  memory_control_unit<word>& mem_ctl,
					  bus_bridge<word>& bridge)
{
	const word size = mem_src.size();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		lobus_if[0].send_single_read_request(adr);
		int flag = 0;
		do {
			cerr << "-- clock" << global_clock << " --" << endl;
			cerr << "in:  lobus: " << lobus_if[0] << endl
				 << "     shbus: " << shbus_if << endl
				 << "     exbus: " << exbus_if << endl
				 << "     bridg: " << bridge << endl
				 << "     cache: " << cache_ctl[0] << endl;
			cache_ctl[0].clock_in();
			bridge.clock_in();
			mem_ctl.clock_in();
			cerr << "out: lobus: " << lobus_if[0] << endl
				 << "     shbus: " << shbus_if << endl
				 << "     exbus: " << exbus_if << endl
				 << "     bridg: " << bridge << endl
				 << "     cache: " << cache_ctl[0] << endl;
			flag = (lobus_if[0].is_ready() &&
					shbus_if.is_ready() &&
					exbus_if.is_ready() &&
					!shbus_if.is_owned());
			if (lobus_if[0].is_single_read_data()) {
				mem_pu[0].write(adr, lobus_if[0].data());
				lobus_if[0].clear();
			}
			cache_ctl[0].clock_out();
			bridge.clock_out();
			mem_ctl.clock_out();
			global_clock++;
		} while (!flag);
		cerr << cache_buf[0] << endl;
	}
	for (int i = 0; i < 10; i++) {
		cerr << "-- clock" << global_clock << " --" << endl;
		cerr << "in:  lobus: " << lobus_if[0] << endl
			 << "     shbus: " << shbus_if << endl
			 << "     exbus: " << exbus_if << endl
			 << "     cache: " << cache_ctl[0] << endl;
		cache_ctl[0].clock_in();
		bridge.clock_in();
		mem_ctl.clock_in();
		cerr << "out: lobus: " << lobus_if[0] << endl
			 << "     shbus: " << shbus_if << endl
			 << "     exbus: " << exbus_if << endl
			 << "     cache: " << cache_ctl[0] << endl << endl;
		cache_ctl[0].clock_out();
		bridge.clock_out();
		mem_ctl.clock_out();
		global_clock++;
	}
}

void single_write_test(cache_ctl_t cache_ctl[],
					   cache_t cache_buf[],
					   bus_port<word> lobus_if[],
					   bus_port<word>& exbus_if,
					   bus_port<bus_data_t>& shbus_if,
					   mapped_memory<word, word>& mem_src,
					   mapped_memory<word, word> mem_pu[],
					   memory_control_unit<word>& mem_ctl,
					   bus_bridge<word>& bridge)
{
	const word size = mem_src.size();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		mem_pu[0].write(adr, adr + 1);
	}
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		lobus_if[0].send_single_write_request(adr);
		int flag = 0;
		do {
			cerr << "-- clock" << global_clock << " --" << endl;
			cerr << "in:  lobus: " << lobus_if[0] << endl
				 << "     shbus: " << shbus_if << endl
				 << "     exbus: " << exbus_if << endl
				 << "     bridg: " << bridge << endl
				 << "     cache: " << cache_ctl[0] << endl;
			cache_ctl[0].clock_in();
			bridge.clock_in();
			mem_ctl.clock_in();
			cerr << "out: lobus: " << lobus_if[0] << endl
				 << "     shbus: " << shbus_if << endl
				 << "     exbus: " << exbus_if << endl
				 << "     bridg: " << bridge << endl
				 << "     cache: " << cache_ctl[0] << endl << endl;
			flag = (lobus_if[0].is_ready() &&
					shbus_if.is_ready() &&
					exbus_if.is_ready() &&
					!shbus_if.is_owned());
			if (lobus_if[0].is_single_write_ack()) {
				lobus_if[0].send_single_write_data(mem_pu[0].read(adr));
			} else if (lobus_if[0].is_single_write_data()) {
				lobus_if[0].clear();
			}
			cache_ctl[0].clock_out();
			bridge.clock_out();
			mem_ctl.clock_out();
			global_clock++;
		} while (!flag);
		cerr << cache_buf[0] << endl;
	}
	for (int i = 0; i < 10; i++) {
		cerr << "-- clock" << global_clock << " --" << endl;
		cerr << "in:  lobus: " << lobus_if[0] << endl
			 << "     shbus: " << shbus_if << endl
			 << "     exbus: " << exbus_if << endl
			 << "     cache: " << cache_ctl[0] << endl;
		cache_ctl[0].clock_in();
		bridge.clock_in();
		mem_ctl.clock_in();
		cerr << "out: lobus: " << lobus_if[0] << endl
			 << "     shbus: " << shbus_if << endl
			 << "     exbus: " << exbus_if << endl
			 << "     cache: " << cache_ctl[0] << endl << endl;
		cache_ctl[0].clock_out();
		bridge.clock_out();
		mem_ctl.clock_out();
		global_clock++;
	}
}

void multi_read_test(cache_ctl_t cache_ctl[],
					 cache_t cache_buf[],
					 bus_port<word> lobus_if[],
					 bus_port<word>& exbus_if,
					 bus_port<bus_data_t>& shbus_if,
					 mapped_memory<word, word>& mem_src,
					 mapped_memory<word, word> mem_pu[],
					 memory_control_unit<word>& mem_ctl,
					 bus_bridge<word>& bridge)
{
	const word size = mem_src.size();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		for (int i = 0; i < punum; i++) {
			lobus_if[i].send_single_read_request(adr);
			int flag = 0, j;
			do {
				cerr << "-- clock" << global_clock << " --" << endl;
				cerr << "in:  lobus" << i << ": " << lobus_if[i] << endl
					 << "     shbus:  " << shbus_if << endl
					 << "     exbus:  " << exbus_if << endl
					 << "     bridge: " << bridge << endl;
				for (j = 0; j < punum; j++) {
					cerr << "     cache" << j << ": "
						 << cache_ctl[j] << endl;
				}
				for (j = 0; j < punum; j++) cache_ctl[j].clock_in();
				bridge.clock_in();
				mem_ctl.clock_in();
				cerr << "out: lobus" << i << ": " << lobus_if[i] << endl
					 << "     shbus:  " << shbus_if << endl
					 << "     exbus:  " << exbus_if << endl
					 << "     bridge: " << bridge << endl;
				for (j = 0; j < punum; j++) {
					cerr << "     cache" << j << ": "
						 << cache_ctl[j] << endl;
				}
				cerr << endl;
				flag = (lobus_if[i].is_ready() &&
						shbus_if.is_ready() &&
						exbus_if.is_ready() &&
						!shbus_if.is_owned());
				if (lobus_if[i].is_single_read_data()) {
					mem_pu[i].write(adr, lobus_if[i].data());
					lobus_if[i].clear();
				}
				for (j = 0; j < punum; j++) cache_ctl[j].clock_out();
				bridge.clock_out();
				mem_ctl.clock_out();
				global_clock++;
			} while (!flag);
			for (j = 0; j < punum; j++) {
				cerr << "cache" << j << ':' << endl
					 << cache_buf[j] << endl;
			}
		}
	}
	for (int i = 0; i < 10; i++) {
		cerr << "-- clock" << global_clock << " --" << endl;
		cerr << "in:  ";
		for (int j = 0; j < punum; j++) {
			if (j) cerr << "     ";
			cerr << "lobus" << j << ": " << lobus_if[j] << endl;
		}
		cerr << "     shbus:  " << shbus_if << endl
			 << "     exbus:  " << exbus_if << endl
			 << "     bridge: " << bridge << endl;
		for (int j = 0; j < punum; j++) {
			cerr << "     cache" << j << ": "
				 << cache_ctl[j] << endl;
		}
		for (int j = 0; j < punum; j++) cache_ctl[j].clock_in();
		bridge.clock_in();
		mem_ctl.clock_in();
		cerr << "out: ";
		for (int j = 0; j < punum; j++) {
			if (j) cerr << "     ";
			cerr << "lobus" << j << ": " << lobus_if[j] << endl;
		}
		cerr << "     shbus:  " << shbus_if << endl
			 << "     exbus:  " << exbus_if << endl
			 << "     bridge: " << bridge << endl;
		for (int j = 0; j < punum; j++) {
			cerr << "     cache" << j << ": "
				 << cache_ctl[j] << endl;
		}
		for (int j = 0; j < punum; j++) cache_ctl[j].clock_out();
		bridge.clock_out();
		mem_ctl.clock_out();
		global_clock++;
	}
}

void multi_write_test(cache_ctl_t cache_ctl[],
					  cache_t cache_buf[],
					  bus_port<word> lobus_if[],
					  bus_port<word>& exbus_if,
					  bus_port<bus_data_t>& shbus_if,
					  mapped_memory<word, word>& mem_src,
					  mapped_memory<word, word> mem_pu[],
					  memory_control_unit<word>& mem_ctl,
					  bus_bridge<word>& bridge)
{
	const word size = mem_src.size();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		for (int i = 0; i < punum; i++) mem_pu[i].write(adr, adr + i + 1);
	}
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		for (int i = 0; i < punum; i++) {
			lobus_if[i].send_single_write_request(adr);
			int flag = 0, j;
			do {
				cerr << "-- clock" << global_clock << " --" << endl;
				cerr << "in:  lobus" << i << ": " << lobus_if[i] << endl
					 << "     shbus:  " << shbus_if << endl
					 << "     exbus:  " << exbus_if << endl
					 << "     bridge: " << bridge << endl;
				for (j = 0; j < punum; j++) {
					cerr << "     cache" << j << ": "
						 << cache_ctl[j] << endl;
				}
				cerr << "     mem:    " << mem_ctl << endl;
				for (j = 0; j < punum; j++) cache_ctl[j].clock_in();
				bridge.clock_in();
				mem_ctl.clock_in();
				cerr << "out: lobus" << i << ": " << lobus_if[i] << endl
					 << "     shbus:  " << shbus_if << endl
					 << "     exbus:  " << exbus_if << endl
					 << "     bridge: " << bridge << endl;
				for (j = 0; j < punum; j++) {
					cerr << "     cache" << j << ": "
						 << cache_ctl[j] << endl;
				}
				cerr << "     mem:    " << mem_ctl << endl << endl;
				flag = (lobus_if[i].is_ready() &&
						shbus_if.is_ready() &&
						exbus_if.is_ready() &&
						!shbus_if.is_owned());
				if (lobus_if[i].is_single_write_ack()) {
					lobus_if[i].send_single_write_data(mem_pu[i].read(adr));
				} else if (lobus_if[i].is_single_write_data()) {
					lobus_if[i].clear();
				}
				for (j = 0; j < punum; j++) cache_ctl[j].clock_out();
				bridge.clock_out();
				mem_ctl.clock_out();
				global_clock++;
			} while (!flag);
			for (j = 0; j < punum; j++) {
				cerr << "cache" << j << ':' << endl
					 << cache_buf[j] << endl;
			}
		}
	}
	for (int i = 0; i < 10; i++) {
		cerr << "-- clock" << global_clock << " --" << endl;
		cerr << "in:  ";
		for (int j = 0; j < punum; j++) {
			if (j) cerr << "     ";
			cerr << "lobus" << j << ": " << lobus_if[j] << endl;
		}
		cerr << "     shbus:  " << shbus_if << endl
			 << "     exbus:  " << exbus_if << endl
			 << "     bridge: " << bridge << endl;
		for (int j = 0; j < punum; j++) {
			cerr << "     cache" << j << ": "
				 << cache_ctl[j] << endl;
		}
		for (int j = 0; j < punum; j++) cache_ctl[j].clock_in();
		bridge.clock_in();
		mem_ctl.clock_in();
		cerr << "out: ";
		for (int j = 0; j < punum; j++) {
			if (j) cerr << "     ";
			cerr << "lobus" << j << ": " << lobus_if[j] << endl;
		}
		cerr << "     shbus:  " << shbus_if << endl
			 << "     exbus:  " << exbus_if << endl
			 << "     bridge: " << bridge << endl;
		for (int j = 0; j < punum; j++) {
			cerr << "     cache" << j << ": "
				 << cache_ctl[j] << endl;
		}
		for (int j = 0; j < punum; j++) cache_ctl[j].clock_out();
		bridge.clock_out();
		mem_ctl.clock_out();
		global_clock++;
	}
}

