/*
 * <<< shared_bus_interface_unit_test.cc >>>
 *
 * --- Test program for shared_bus_interface_unit class
 *     Copyright (C) 2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <iostream>
#include "bus_port.h"
#include "memory_control_unit.h"
#include "mapped_memory.h"
#include "shared_bus_interface_unit.h"

typedef unsigned int word;

static void single_read_test(mapped_memory<word, word>&,
							 mapped_memory<word, word>&, 
							 memory_control_unit<word>&,
							 shared_bus_interface_unit<word>&,
							 bus_port<word>&, bus_port<word>&);
static void single_write_test(mapped_memory<word, word>&,
							  mapped_memory<word, word>&, 
							  memory_control_unit<word>&,
							  shared_bus_interface_unit<word>&,
							  bus_port<word>&, bus_port<word>&);
static void multi_read_test(mapped_memory<word, word>&,
							mapped_memory<word, word>&, 
							memory_control_unit<word>&,
							shared_bus_interface_unit<word>&,
							bus_port<word>&, bus_port<word>&, size_t);
static void multi_write_test(mapped_memory<word, word>&,
							 mapped_memory<word, word>&, 
							 memory_control_unit<word>&,
							 shared_bus_interface_unit<word>&,
							 bus_port<word>&, bus_port<word>&, size_t);

int main(void)
{
	const word top = 0, size = 0x20;
	const int multi_count = 4;
	shared_bus_interface_unit<word> shbus_if_unit;
	bus_port<word> lobus_if, shbus_if;
	mapped_memory<word, word> mem_a(top, size), mem_b(top, size);
	memory_control_unit<word> mem_ctl;
	shbus_if_unit.connect_local_bus_port(lobus_if);
	shbus_if_unit.connect_shared_bus_port(shbus_if);
	shbus_if_unit.connect_shared_bus_port(mem_ctl);
	shbus_if_unit.set_valid_address(mem_a.top(), mem_a.size());
	mem_ctl.connect_memory(mem_a);

	cout << "=== single read test" << endl;
	cerr << "=== single read test" << endl;
	single_read_test(mem_a, mem_b, mem_ctl, shbus_if_unit, lobus_if, shbus_if);
	cout << "src:" << endl << mem_a << endl
		 << "dst:" << endl << mem_b << endl;

	cout << "=== single write test" << endl;
	cerr << "=== single write test" << endl;
	single_write_test(mem_a, mem_b, mem_ctl, shbus_if_unit, lobus_if,
					  shbus_if);
	cout << "src:" << endl << mem_b << endl
		 << "dst:" << endl << mem_a << endl;

	cout << "=== multi read test" << endl;
	cerr << "=== multi read test" << endl;
	multi_read_test(mem_a, mem_b, mem_ctl, shbus_if_unit, lobus_if, shbus_if,
					multi_count);
	cout << "src:" << endl << mem_b << endl
		 << "dst:" << endl << mem_a << endl;

	cout << "=== multi write test" << endl;
	cerr << "=== multi write test" << endl;
	multi_write_test(mem_a, mem_b, mem_ctl, shbus_if_unit, lobus_if, shbus_if,
					 multi_count);
	cout << "src:" << endl << mem_b << endl
		 << "dst:" << endl << mem_a << endl;

	return 0;
}

void single_read_test(mapped_memory<word, word>& mem_a,
					  mapped_memory<word, word>& mem_b, 
					  memory_control_unit<word>& mem_ctl,
					  shared_bus_interface_unit<word>& shbus_if_unit,
					  bus_port<word>& lobus_if, bus_port<word>& shbus_if)
{
	const word size = mem_a.size();
	mem_a.clear();
	mem_b.clear();
	for (word adr = 0; adr < word(size); adr += sizeof(word)) {
		mem_a.write(adr, adr);
	}
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		lobus_if.send_single_read_request(adr);
		int flag = 0;
		do {
			cerr << "in:  lobus: " << lobus_if << endl
				 << "     shbus: " << shbus_if << endl
				 << "     b i/f: " << shbus_if_unit << endl;
			shbus_if_unit.clock_in();
			mem_ctl.clock_in();
			cerr << "out: lobus: " << lobus_if << endl
				 << "     shbus: " << shbus_if << endl
				 << "     b i/f: " << shbus_if_unit << endl;
			if (lobus_if.is_single_read_data()) {
				mem_b.write(adr, lobus_if.data());
				lobus_if.clear();
			}
			flag = (lobus_if.is_ready() && shbus_if.is_ready());
			shbus_if_unit.clock_out();
			mem_ctl.clock_out();
		} while (!flag);
	}
}

void single_write_test(mapped_memory<word, word>& mem_a,
					   mapped_memory<word, word>& mem_b, 
					   memory_control_unit<word>& mem_ctl,
					   shared_bus_interface_unit<word>& shbus_if_unit,
					   bus_port<word>& lobus_if, bus_port<word>& shbus_if)
{
	const word size = mem_a.size();
	mem_a.clear();
	mem_b.clear();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		mem_b.write(adr, adr);
	}
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		lobus_if.send_single_write_request(adr);
		int flag = 0;
		do {
			cerr << "in:  lobus: " << lobus_if << endl
				 << "     shbus: " << shbus_if << endl
				 << "     b i/f: " << shbus_if_unit << endl;
			shbus_if_unit.clock_in();
			mem_ctl.clock_in();
			int ack_flag = lobus_if.is_single_write_ack();
			int data_flag = lobus_if.is_single_write_data();
			flag = (lobus_if.is_ready() && shbus_if.is_ready());
			cerr << "out: lobus: " << lobus_if << endl
				 << "     shbus: " << shbus_if << endl
				 << "     b i/f: " << shbus_if_unit << endl;
			if (ack_flag) {
				lobus_if.send_single_write_data(mem_b.read(adr));
			} else if (data_flag) {
				lobus_if.clear();
			}
			shbus_if_unit.clock_out();
			mem_ctl.clock_out();
		} while (!flag);
	}
}

void multi_read_test(mapped_memory<word, word>& mem_a,
					 mapped_memory<word, word>& mem_b, 
					 memory_control_unit<word>& mem_ctl,
					 shared_bus_interface_unit<word>& shbus_if_unit,
					 bus_port<word>& lobus_if, bus_port<word>& shbus_if,
					 size_t multi_count)
{
	const word size = mem_a.size();
	mem_a.clear();
	mem_b.clear();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		mem_a.write(adr, adr);
	}
	for (word adr = 0; adr < size; adr += sizeof(word) * multi_count) {
		lobus_if.send_multi_read_request(adr, multi_count);
		int flag = 0;
		do {
			cerr << "in:  lobus: " << lobus_if << endl
				 << "     shbus: " << shbus_if << endl
				 << "     b i/f: " << shbus_if_unit << endl;
			shbus_if_unit.clock_in();
			mem_ctl.clock_in();
			cerr << "out: lobus: " << lobus_if << endl
				 << "     shbus: " << shbus_if << endl
				 << "     b i/f: " << shbus_if_unit << endl;
			if (lobus_if.is_multi_read_data()) {
				mem_b.write(adr + lobus_if.packet_number() * sizeof(word),
					lobus_if.data());
				if (lobus_if.packet_number() == multi_count - 1) {
					lobus_if.clear();
				}
			}
			flag = (lobus_if.is_ready() && shbus_if.is_ready());
			shbus_if_unit.clock_out();
			mem_ctl.clock_out();
		} while (!flag);
	}
}

void multi_write_test(mapped_memory<word, word>& mem_a,
					  mapped_memory<word, word>& mem_b, 
					  memory_control_unit<word>& mem_ctl,
					  shared_bus_interface_unit<word>& shbus_if_unit,
					  bus_port<word>& lobus_if, bus_port<word>& shbus_if,
					  size_t multi_count)
{
	const word size = mem_a.size();
	mem_a.clear();
	mem_b.clear();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		mem_b.write(adr, adr);
	}
	for (word adr = 0; adr < size; adr += sizeof(word) * multi_count) {
		lobus_if.send_multi_write_request(adr, multi_count);
		bool flag = false;
		size_t i = 0;
		do {
			cerr << "in:  lobus: " << lobus_if << endl
				 << "     shbus: " << shbus_if << endl
				 << "     b i/f: " << shbus_if_unit << endl;
			shbus_if_unit.clock_in();
			mem_ctl.clock_in();
			bool ack_flag = lobus_if.is_multi_write_ack();
			flag = (lobus_if.is_ready() && shbus_if.is_ready());
			cerr << "out: lobus: " << lobus_if << endl
				 << "     shbus: " << shbus_if << endl
				 << "     b i/f: " << shbus_if_unit << endl;
			if (ack_flag || (i > 0 && i < multi_count)) {
				lobus_if.send_multi_write_data(
					mem_b.read(adr + sizeof(word) * i), multi_count, i);
				i++;
			} else if (i == multi_count) {
				lobus_if.clear();
			}
			shbus_if_unit.clock_out();
			mem_ctl.clock_out();
		} while (!flag);
	}
}
