/*
 * <<< main.cc >>>
 *
 * --- Shared bus architecture simulator 'isis_sharedbus'
 *     Copyright (C) 1998-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <iostream>
#include <fstream>
#include "isis.h"
#include "argument_parser.h"
#include "gdb_stub.h"
#include "cui_debugger.h"
#include "gui_debugger.h"
#include "shared_bus_architecture.h"

const r3000_word local_memory_address  = 0x00000000;
const r3000_word local_memory_size	   = 0x20000000;
const r3000_word shared_memory_address = 0xa0000000;
const r3000_word shared_memory_size	   = 0x10000000;
const r3000_word sync_memory_address   = 0xb0000000;
const r3000_word sync_memory_size	   = 0x08000000;

static const char* to_basename(const char* s)
{
	const char* head = s;
	while (*s != '\0') {
		if (*s == '/') head = s + 1;
		s++;
	}
	return head;
}

void usage(const char* name)
{
	cout << "usage: " << name
		 << " <target> [options] [-- [target options]]" << endl
		 << "options:" << endl
		 << "  -c,    --clock         print execution time in clock."  << endl
		 << "  -d<n>, --delay=<n>     set shared memory access delay." << endl
		 << "  -g,    --gdb           use gdb mode."				   << endl
		 << "  -h,    --help          print this message."			   << endl
		 << "  -m,    --monitor       use monitor mode."			   << endl
		 << "  -p<n>, --processor=<n> set number of processors."	   << endl
		 << "  -v,    --verbose       verbosely output."			   << endl
		 << "  --udp                  use gdb mode with udp."		   << endl
		 << "  --version              print version information."	   << endl
		 << "  --stdin=<file>         set standard input file."		   << endl
		 << "  --stdout=<file>        set standard output file."	   << endl
		 << "  --stderr=<file>        set standard error file."		   << endl;
}

void version(const char* name)
{
	cout << name << " - " << isis::version_string() << endl;
}

int main(int, char** argv)
{
	typedef shared_bus_architecture architecture_type;
	typedef shared_bus_element processing_element_type;
	bool clock_flag = false, cui_monitor_flag = false,
		 gui_monitor_flag = false,gdb_flag = false,
		 udp_flag = false, verbose_flag = false;
	size_t punum = 1, delay = 0;
	argument_parser arg((const char* const*)(argv + 1));
	if (arg.defined('h') || arg.defined("help"))
		{ usage(to_basename(argv[0])); return 0; }
	if (arg.defined("version"))
		{ version(to_basename(argv[0])); return 0; }
	if (arg.defined('c') || arg.defined("clock"))
		clock_flag = true;
	if (arg.defined('g') || arg.defined("gdb"))
		gdb_flag = true;
	if (arg.defined("gui"))
		gui_monitor_flag = true;
	if (arg.defined('m') || arg.defined("monitor"))
		cui_monitor_flag = true;
	if (arg.defined("udp"))
		udp_flag = true;
	if (arg.defined('v') || arg.defined("verbose"))
		verbose_flag = clock_flag = true;
	if (arg['d'] != NULL)		  delay = atoi(arg['d']);
	if (arg["delay"] != NULL)	  delay = atoi(arg["delay"]);
	if (arg['p'] != NULL)		  punum = atoi(arg['p']);
	if (arg["processor"] != NULL) punum = atoi(arg["processor"]);
	architecture_type com(punum);
	{
		for (size_t i = 0; i < com.processor_number(); i++) {
			processing_element_type& pe(com.processing_element(i));
			pe.set_local_memory_area(local_memory_address, local_memory_size);
			pe.set_file_table_size(16);
		}
		com.set_shared_memory_area(shared_memory_address, shared_memory_size);
		com.set_sync_memory_area(sync_memory_address, sync_memory_size);
	}
	ifstream fin;
	ofstream fout, ferr;
	if (arg["stdin"] != NULL) {
		fin.open(arg["stdin"], ios::in);
		if (!fin) {
			cerr << "open error: " << arg["stdin"] << endl;
			return 1;
		}
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_input_stream(fin);
		}
	} else {
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_input_stream(cin);
		}
	}
	if (arg["stdout"] != NULL) {
		fout.open(arg["stdout"], ios::out);
		if (!fout) {
			cerr << "open error: " << arg["stdout"] << endl;
			return 1;
		}
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_output_stream(cout);
		}
	} else {
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_output_stream(cout);
		}
	}
	if (arg["stderr"] != NULL) {
		ferr.open(arg["stderr"], ios::out);
		if (!ferr) {
			cerr << "open error: " << arg["stderr"] << endl;
			return 1;
		}
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_error_stream(cerr);
		}
	} else {
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_error_stream(cerr);
		}
	}
	if (arg.argument()[0] == NULL) {
		cerr << "No executable file specified." << endl;
		return 1;
	}
	{
		for (size_t i = 0; i < com.processor_number(); i++) {
			processing_element_type& pe(com.processing_element(i));
			if (!pe.load(arg.argument()[0])) {
				cerr << arg.argument()[0]
					 << ": No such file or directory." << endl;
				return 1;
			}
			pe.set_commandline_argument(arg.argument());
		}
	}
	if (delay > 0) {
		com.set_shared_memory_read_wait(delay);
		com.set_shared_memory_write_wait(delay);
		com.set_sync_memory_read_wait(delay);
		com.set_sync_memory_write_wait(delay);
	}
	if (gdb_flag) {
		gdb_stub<r3000_word> stub;
		stub.set_multiprocessor(com);
		stub.set_debug_output(cerr);
		if (udp_flag) stub.use_udp();
		if (verbose_flag) stub.set_verbose();
		stub.service();
		return 0;
	} else if (cui_monitor_flag) {
		cui_debugger<processing_element_type> mon;
		mon.set(com);
		mon.set_prompt("> ");
		mon.interactive_mode();
		return 0;
	} else if (gui_monitor_flag) {
		gui_debugger<processing_element_type> mon;
		mon.set(com);
		mon.interactive_mode();
		return 0;
	} else {
		processing_element_type& pe(com.processing_element(0));
		size_t count = 0;
		while (!pe.is_halt()) {
			if (verbose_flag) {
				cerr << "--- clk:" << pe.timer_clock_value() << " ---" << endl;
				cerr << "in:" << endl << com << endl;
			}
			if ((++count & 0x3ff == 0) && com.is_bus_error()) {
				size_t i;
				for (i = 0; i < com.processor_number(); i++) {
					if (com.processing_element(i).is_bus_error()) break;
				}
				processing_element_type& pe(com.processing_element(i));
				cout << "pu" << i << ": " << hex
					 << "bus error(I:0x" << pe.processor().program_counter()
					 << ", D:0x" << pe.bus_error_address() << ')'
					 << dec << endl;
				break;
			}
			com.clock_in();
			if (verbose_flag) {
				cerr << "out:" << endl << com << endl;
			}
			com.clock_out();
		}
		if (clock_flag) cout << "clock: " << pe.timer_clock_value() << endl;
		return pe.commandline_status();
	}
	/* not reached */
}
