/* -*- C++ -*-
 *
 * <<< gui_debugger.h >>>
 *
 * --- Graphical user interface debugger class 'gui_debugger'
 *     Copyright (C) 2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef GUI_DEBUGGER_H
#define GUI_DEBUGGER_H 1

#include "isis_config.h"

#if USE_GUI

#include <strstream>
#include <gtk/gtk.h>
#include "debugger.h"

template <class T>
class gui_debugger : public debugger<T>
{
private:
	typedef gui_debugger<T> thisclass;
	typedef debugger<T> inherited;
public:
	typedef typename inherited::address_type address_type;
	typedef typename inherited::data_type data_type;
	typedef typename inherited::char_type char_type;
	typedef typename inherited::size_type size_type;
	typedef typename inherited::processor_type processor_type;
	typedef typename inherited::memory_map_type memory_map_type;
	typedef typename inherited::processing_element_type processing_element_type;
	typedef typename inherited::multiprocessor_type multiprocessor_type;
	static const size_t sizeof_data_type = inherited::sizeof_data_type;
private:
	static const char credits_message[];
	static const GtkItemFactoryEntry menu_items[];
	static thisclass* active_debugger;
	static GtkWidget* main_window;
	static GtkWidget* processor_frame;
	static GtkWidget* screen_frame;
	static GtkWidget* disassemble_frame;
	static GtkWidget* memory_frame;
	static GtkWidget* status_frame;
	static GtkWidget* about_window;
	static GtkAccelGroup* main_accel_group;
	static vector<GtkWidget*> processor_register_entry_widget;
	static vector<GtkWidget*> processor_pipeline_entry_widget;
	static GtkWidget* processor_status_entry_widget;
	static GtkWidget* screen_text_widget;
	static GtkWidget* disassemble_clist_widget;
	static GtkWidget* memory_clist_widget;
	static GtkWidget* status_clock_label_widget;
	static vector<data_type> processor_register_buf;
	char* screen_standard_output_stream_buf;
	char* screen_standard_error_stream_buf;
	ostrstream screen_standard_output_stream;
	ostrstream screen_standard_error_stream;
	static address_type disassemble_clist_address;
	static size_t disassemble_clist_size;
	static address_type memory_clist_address;
	static size_t memory_clist_size;
	static void main(int, char**);
	static void create_processor_frame(void);
	static void create_screen_frame(void);
	static void create_disassemble_frame(void);
	static void create_memory_frame(void);
	static void create_status_frame(void);
	static void create_about_window(void);
	static void update_processor_frame(void);
	static void update_screen_frame(void);
	static void update_status_frame(void);
	static void update_disassemble_frame(void);
	static void update_memory_frame(void);
	static void show_about_window(void);
	static void step_callback(void);
	static void continue_callback(void);
public:
	gui_debugger(void);
	gui_debugger(const gui_debugger<T>&);
	virtual ~gui_debugger();
	void interactive_mode(void);
};

template <class T>
const char gui_debugger<T>::credits_message[] =
	"ISIS - Multiprocessor Simulator Library\n"
	"\n"
	"Copyright (C) 1995-2000 Amano Lab., Keio University.\n"
	"All rights reserved.";

template <class T>
const GtkItemFactoryEntry gui_debugger<T>::menu_items[] = {
	{ "/_File", NULL,
		NULL, 0, "<Branch>" },
	{ "/File/_Breakpoints Window", NULL,
		NULL, 0, "<Item>" },
	{ "/File/sep1", NULL,
		NULL, 0, "<Separator>" },
	{ "/File/_Quit", "<control>Q",
		gtk_main_quit, 0, "<Item>" },
	{ "/_Run", NULL,
		NULL, 0, "<Branch>" },
	{ "/Run/_Step", "S",
		gui_debugger<T>::step_callback, 0, "<Item>" },
	{ "/Run/_Go", "G",
		gui_debugger<T>::continue_callback, 0, "<Item>" },
	{ "/_Help", NULL,
		NULL, 0, "<LastBranch>" },
	{ "/Help/_About", "<control>A",
		show_about_window, 0, "<Item>" },
};

template <class T> gui_debugger<T>* gui_debugger<T>::active_debugger = NULL;
template <class T> GtkWidget* gui_debugger<T>::main_window;
template <class T> GtkWidget* gui_debugger<T>::processor_frame;
template <class T> GtkWidget* gui_debugger<T>::screen_frame;
template <class T> GtkWidget* gui_debugger<T>::disassemble_frame;
template <class T> GtkWidget* gui_debugger<T>::memory_frame;
template <class T> GtkWidget* gui_debugger<T>::status_frame;
template <class T> GtkWidget* gui_debugger<T>::about_window = NULL;
template <class T> GtkAccelGroup* gui_debugger<T>::main_accel_group;
template <class T>
	vector<GtkWidget*> gui_debugger<T>::processor_register_entry_widget;
template <class T>
	vector<GtkWidget*> gui_debugger<T>::processor_pipeline_entry_widget;
template <class T> GtkWidget* gui_debugger<T>::processor_status_entry_widget;
template <class T> GtkWidget* gui_debugger<T>::screen_text_widget;
template <class T> GtkWidget* gui_debugger<T>::disassemble_clist_widget;
template <class T> GtkWidget* gui_debugger<T>::memory_clist_widget;
template <class T> GtkWidget* gui_debugger<T>::status_clock_label_widget;
template <class T>
	vector<gui_debugger<T>::data_type> gui_debugger<T>::processor_register_buf;
template <class T>
	gui_debugger<T>::address_type gui_debugger<T>::disassemble_clist_address;
template <class T> size_t gui_debugger<T>::disassemble_clist_size = 256;
template <class T>
	gui_debugger<T>::address_type gui_debugger<T>::memory_clist_address;
template <class T> size_t gui_debugger<T>::memory_clist_size = 256;

template <class T>
gui_debugger<T>::gui_debugger(void)
	: inherited(),
	  screen_standard_output_stream_buf(new char[1024]),
	  screen_standard_error_stream_buf(new char[1024]),
	  screen_standard_output_stream(screen_standard_output_stream_buf,
	  	sizeof(screen_standard_output_stream_buf)),
	  screen_standard_error_stream(screen_standard_error_stream_buf,
	  	sizeof(screen_standard_error_stream_buf))
{}

template <class T>
gui_debugger<T>::gui_debugger(const gui_debugger<T>& a)
	: inherited(a),
	  screen_standard_output_stream_buf(new char[1024]),
	  screen_standard_error_stream_buf(new char[1024]),
	  screen_standard_output_stream(screen_standard_output_stream_buf,
	  	sizeof(screen_standard_output_stream_buf)),
	  screen_standard_error_stream(screen_standard_error_stream_buf,
	  	sizeof(screen_standard_error_stream_buf))
{}

template <class T>
gui_debugger<T>::~gui_debugger()
{
	delete[] screen_standard_output_stream_buf;
	delete[] screen_standard_error_stream_buf;
}

template <class T>
void gui_debugger<T>::interactive_mode(void)
{
	if (active_debugger != NULL) {
		cerr << "Sorry, another GUI debugger is running." << endl;
		return;
	}
	active_debugger = this;
	int dummy_argc = 1;
	char* dummy_args[] = { "gui_debugger", NULL };
	main(dummy_argc, dummy_args);
	active_debugger = NULL;
}

template <class T>
void gui_debugger<T>::main(int argc, char** argv)
{
	GtkWidget* top_vbox;
	GtkWidget* menubar;
	GtkWidget* toolbar;
	GtkWidget* main_frame;
	GtkWidget* main_table;
	GtkWidget* right_vpaned;
	GtkWidget* processor_window;
	GtkWidget* screen_window;
	GtkWidget* disassemble_window;
	GtkWidget* memory_window;
	GtkWidget* status_window;

	// initialize gtk+
	// gtk_set_locale();
	gtk_init(&argc, &argv);

	// initialize inner variables
	{
		processor_type& pu(active_debugger->current_pe().processor());
		// registers
		{
			size_t size = 32; // conclete value
			processor_register_entry_widget.resize(size);
			processor_register_buf.resize(size);
			for (size_t i = 0; i < size; i++) {
				processor_register_buf[i] = pu.register_file(i);
			}
		}
		// pipeline
		{
			size_t size = 5; // conclete value
			processor_pipeline_entry_widget.resize(size);
		}
		// disassemble, memory
		disassemble_clist_address = memory_clist_address = pu.program_counter();
	}

	// create top window
	main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(main_window), "destroy",
					   GTK_SIGNAL_FUNC(gtk_main_quit), (void*)"WM destroy");
	gtk_window_set_title(GTK_WINDOW(main_window), "ISIS");
	gtk_window_set_policy(GTK_WINDOW(main_window), FALSE, FALSE, FALSE);
	gtk_container_border_width(GTK_CONTAINER(main_window), 0);

	// create main accel group
	main_accel_group = gtk_accel_group_new();

	// create top vbox
	top_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(main_window), top_vbox);
	gtk_widget_show(top_vbox);

	// create menubar and set short-cut
	{
		GtkItemFactory* item_factory;
		size_t size = sizeof(menu_items) / sizeof(menu_items[0]);
		item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>",
											main_accel_group);
		gtk_item_factory_create_items(item_factory, size,
									  (GtkItemFactoryEntry*)menu_items, NULL);
		menubar = gtk_item_factory_get_widget(item_factory, "<main>");
	}
	gtk_box_pack_start(GTK_BOX(top_vbox), menubar, FALSE, FALSE, 0);
	gtk_widget_show(menubar);
	gtk_window_add_accel_group(GTK_WINDOW(main_window), main_accel_group);

	// create toolbar
	toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_TEXT);
	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),
		"Step", "Step program", NULL, NULL, step_callback, NULL);
	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),
		"Go", "Continue program", NULL, NULL, continue_callback, NULL);
	gtk_box_pack_start(GTK_BOX(top_vbox), toolbar, FALSE, FALSE, 0);
	gtk_widget_show(toolbar);

	// create toolbar separator
	{
		GtkWidget* separator = gtk_hseparator_new();
		gtk_box_pack_start(GTK_BOX(top_vbox), separator, FALSE, FALSE, 0);
		gtk_widget_show(separator);
	}

	// create main frame and table
	main_frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(main_frame), GTK_SHADOW_NONE);
	gtk_container_border_width(GTK_CONTAINER(main_frame), 1);
	gtk_box_pack_start(GTK_BOX(top_vbox), main_frame, TRUE, TRUE, 0);
	gtk_widget_show(main_frame);
	main_table = gtk_table_new(2, 2, FALSE);
	gtk_container_add(GTK_CONTAINER(main_frame), main_table);
	gtk_widget_show(main_table);

	// create right vpaned
	right_vpaned = gtk_vpaned_new();
	gtk_table_attach(GTK_TABLE(main_table), right_vpaned,
					 1, 2, 0, 2,
					 GtkAttachOptions(GTK_EXPAND | GTK_FILL),
					 GtkAttachOptions(GTK_EXPAND | GTK_FILL | GTK_SHRINK),
					 0, 0);
	gtk_widget_show(right_vpaned);

	// create processor window
	processor_window = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(processor_window), GTK_SHADOW_IN);
	gtk_container_border_width(GTK_CONTAINER(processor_window), 3);
	gtk_table_attach(GTK_TABLE(main_table), processor_window,
					 0, 1, 0, 1,
					 GtkAttachOptions(GTK_FILL),
					 GtkAttachOptions(GTK_FILL | GTK_SHRINK), 0, 0);
	gtk_widget_show(processor_window);
	{
		GtkWidget* vbox;
		GtkWidget* label_frame;
		GtkWidget* label;
		vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(processor_window), vbox);
		gtk_widget_show(vbox);
		label_frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(label_frame), GTK_SHADOW_OUT);
		gtk_box_pack_start(GTK_BOX(vbox), label_frame, FALSE, FALSE, 0);
		gtk_widget_show(label_frame);
		label = gtk_label_new("Processor");
		gtk_container_add(GTK_CONTAINER(label_frame), label);
		gtk_widget_show(label);
		create_processor_frame();
		update_processor_frame();
		gtk_box_pack_start(GTK_BOX(vbox), processor_frame, FALSE, FALSE, 0);
		gtk_widget_show(processor_frame);
	}

	// create screen window
	screen_window = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(screen_window), GTK_SHADOW_IN);
	gtk_container_border_width(GTK_CONTAINER(screen_window), 3);
	gtk_table_attach(GTK_TABLE(main_table), screen_window,
					 0, 1, 1, 2,
					 GtkAttachOptions(GTK_FILL),
					 GtkAttachOptions(GTK_EXPAND | GTK_FILL | GTK_SHRINK),
					 0, 0);
	gtk_widget_show(screen_window);
	{
		GtkWidget* vbox;
		GtkWidget* label_frame;
		GtkWidget* label;
		vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(screen_window), vbox);
		gtk_widget_show(vbox);
		label_frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(label_frame), GTK_SHADOW_OUT);
		gtk_box_pack_start(GTK_BOX(vbox), label_frame, FALSE, FALSE, 0);
		gtk_widget_show(label_frame);
		label = gtk_label_new("Screen");
		gtk_container_add(GTK_CONTAINER(label_frame), label);
		gtk_widget_show(label);
		create_screen_frame();
		gtk_box_pack_start(GTK_BOX(vbox), screen_frame, TRUE, TRUE, 0);
		gtk_widget_show(screen_frame);
	}

	// create disassemble window
	disassemble_window = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(disassemble_window), GTK_SHADOW_IN);
	gtk_container_border_width(GTK_CONTAINER(disassemble_window), 3);
	gtk_paned_add1(GTK_PANED(right_vpaned), disassemble_window);
	gtk_widget_show(disassemble_window);
	{
		GtkWidget* vbox;
		GtkWidget* label_frame;
		GtkWidget* label;
		vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(disassemble_window), vbox);
		gtk_widget_show(vbox);
		label_frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(label_frame), GTK_SHADOW_OUT);
		gtk_box_pack_start(GTK_BOX(vbox), label_frame, FALSE, FALSE, 0);
		gtk_widget_show(label_frame);
		label = gtk_label_new("Disassemble");
		gtk_container_add(GTK_CONTAINER(label_frame), label);
		gtk_widget_show(label);
		create_disassemble_frame();
		update_disassemble_frame();
		gtk_box_pack_start(GTK_BOX(vbox), disassemble_frame, TRUE, TRUE, 0);
		gtk_widget_show(disassemble_frame);
	}

	// create memory window
	memory_window = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(memory_window), GTK_SHADOW_IN);
	gtk_container_border_width(GTK_CONTAINER(memory_window), 3);
	gtk_paned_add2(GTK_PANED(right_vpaned), memory_window);
	gtk_widget_show(memory_window);
	{
		GtkWidget* vbox;
		GtkWidget* label_frame;
		GtkWidget* label;
		vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(memory_window), vbox);
		gtk_widget_show(vbox);
		label_frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(label_frame), GTK_SHADOW_OUT);
		gtk_box_pack_start(GTK_BOX(vbox), label_frame, FALSE, FALSE, 0);
		gtk_widget_show(label_frame);
		label = gtk_label_new("Memory");
		gtk_container_add(GTK_CONTAINER(label_frame), label);
		gtk_widget_show(label);
		create_memory_frame();
		update_memory_frame();
		gtk_box_pack_start(GTK_BOX(vbox), memory_frame, TRUE, TRUE, 0);
		gtk_widget_show(memory_frame);
	}

	// create status window
	create_status_frame();
	update_status_frame();
	gtk_box_pack_end(GTK_BOX(top_vbox), status_frame, FALSE, FALSE, 0);
	gtk_widget_show(status_frame);

	// create status separator
	{
		GtkWidget* separator = gtk_hseparator_new();
		gtk_box_pack_end(GTK_BOX(top_vbox), separator, FALSE, FALSE, 0);
		gtk_widget_show(separator);
	}

	// show window and goto event loop
	gtk_widget_show(main_window);
	gtk_main();
}

template <class T>
void gui_debugger<T>::create_processor_frame(void)
{
	const size_t reg_size = processor_register_entry_widget.size();
	const size_t pipe_size = processor_pipeline_entry_widget.size();
	GtkWidget* top_vbox;
	GtkWidget* reg_frame;
	GtkWidget* reg_window;
	GtkWidget* pipe_frame;
	GtkWidget* pipe_window;
	GtkWidget* status_frame;
	GtkWidget* status_window;
	int reg_xsize, reg_ysize, i;

	// calc register window size
	if (reg_size <= 4) {
		reg_xsize = 1;
		reg_ysize = reg_size;
	} else {
		reg_xsize = 4;
		reg_ysize = (reg_size + reg_xsize - 1) / reg_xsize;
	}

	// create top frame and vbox
	processor_frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(processor_frame), GTK_SHADOW_NONE);
	gtk_container_border_width(GTK_CONTAINER(processor_frame), 0);
	top_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(processor_frame), top_vbox);
	gtk_widget_show(top_vbox);

	// create register frame and window
	reg_frame = gtk_frame_new(NULL);
	gtk_container_set_border_width(GTK_CONTAINER(reg_frame), 3);
	gtk_frame_set_shadow_type(GTK_FRAME(reg_frame), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(top_vbox), reg_frame, FALSE, FALSE, 0);
	gtk_widget_show(reg_frame);
	reg_window = gtk_table_new(reg_xsize * 2, reg_ysize + 1, FALSE);
	gtk_container_add(GTK_CONTAINER(reg_frame), reg_window);
	{
		GtkWidget* frame;
		GtkWidget* label;
		frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
		gtk_table_attach(GTK_TABLE(reg_window), frame,
						 0, reg_xsize * 2 + 1, 0, 1,
						 GtkAttachOptions(GTK_FILL),
						 GtkAttachOptions(GTK_FILL | GTK_SHRINK), 0, 0);
		gtk_widget_show(frame);
		label = gtk_label_new("Registers");
		gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
		gtk_misc_set_padding(GTK_MISC(label), 2, 1);
		gtk_container_add(GTK_CONTAINER(frame), label);
		gtk_widget_show(label);
	}
	for (i = 0; i < reg_size; i++) {
		char buf[256];
		GtkWidget* name_frame;
		GtkWidget* name;
		GtkWidget* value;
		int x, y;
		x = i % reg_xsize;
		y = i / reg_xsize;
		{
			ostrstream os(buf, sizeof(buf));
			os << 'r' << setfill('0') << setw(2) << i << ends;
		}
		name_frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(name_frame), GTK_SHADOW_OUT);
		gtk_table_attach(GTK_TABLE(reg_window), name_frame,
						 x * 2, x * 2 + 1, y + 1, y + 2,
						 GtkAttachOptions(GTK_FILL),
						 GtkAttachOptions(GTK_FILL | GTK_SHRINK), 0, 0);
		gtk_widget_show(name_frame);
		name = gtk_label_new(buf);
		gtk_container_add(GTK_CONTAINER(name_frame), name);
		gtk_widget_show(name);
		value = gtk_entry_new();
		gtk_entry_set_editable(GTK_ENTRY(value), FALSE);
		gtk_widget_set_usize(GTK_WIDGET(value), 80, 21);
		{
			ostrstream os(buf, sizeof(buf));
			os << hex << setfill('0') << setw(8)
			   << processor_register_buf[i] << ends;
		}
		gtk_entry_set_text(GTK_ENTRY(value), buf);
		gtk_table_attach(GTK_TABLE(reg_window), value,
						 x * 2 + 1, x * 2 + 2, y + 1, y + 2,
						 GtkAttachOptions(GTK_FILL),
						 GtkAttachOptions(GTK_FILL | GTK_SHRINK), 0, 0);
		gtk_widget_show(value);
		processor_register_entry_widget[i] = value;
	}
	gtk_widget_show(reg_window);

	// create pipeline frame and window
	pipe_frame = gtk_frame_new(NULL);
	gtk_container_set_border_width(GTK_CONTAINER(pipe_frame), 3);
	gtk_frame_set_shadow_type(GTK_FRAME(pipe_frame), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(top_vbox), pipe_frame, FALSE, FALSE, 0);
	gtk_widget_show(pipe_frame);
	pipe_window = gtk_table_new(2, pipe_size + 1, FALSE);
	gtk_container_add(GTK_CONTAINER(pipe_frame), pipe_window);
	{
		GtkWidget* frame;
		GtkWidget* label;
		frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
		gtk_table_attach(GTK_TABLE(pipe_window), frame,
						 0, 2, 0, 1,
						 GtkAttachOptions(GTK_FILL),
						 GtkAttachOptions(GTK_FILL | GTK_SHRINK), 0, 0);
		gtk_widget_show(frame);
		label = gtk_label_new("Instruction Pipeline");
		gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
		gtk_misc_set_padding(GTK_MISC(label), 2, 1);
		gtk_container_add(GTK_CONTAINER(frame), label);
		gtk_widget_show(label);
	}
	for (i = 0; i < pipe_size; i++) {
		char buf[256];
		GtkWidget* name_frame;
		GtkWidget* name;
		GtkWidget* value;
		{
			ostrstream os(buf, sizeof(buf));
			switch (i) {
			case 0: os << "if" << ends; break;
			case 1: os << "rd" << ends; break;
			case 2: os << "ex" << ends; break;
			case 3: os << "mm" << ends; break;
			case 4: os << "wb" << ends; break;
			default: break;
			}
		}
		name_frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(name_frame), GTK_SHADOW_OUT);
		gtk_table_attach(GTK_TABLE(pipe_window), name_frame, 0, 1, i + 1, i + 2,
						 GtkAttachOptions(GTK_FILL),
						 GtkAttachOptions(GTK_FILL | GTK_SHRINK), 0, 0);
		gtk_widget_show(name_frame);
		name = gtk_label_new(buf);
		gtk_container_add(GTK_CONTAINER(name_frame), name);
		gtk_widget_show(name);
		value = gtk_entry_new();
		gtk_entry_set_editable(GTK_ENTRY(value), FALSE);
		gtk_table_attach(GTK_TABLE(pipe_window), value, 1, 2, i + 1, i + 2,
						 GtkAttachOptions(GTK_EXPAND | GTK_FILL),
						 GtkAttachOptions(GTK_EXPAND | GTK_FILL | GTK_SHRINK),
						 0, 0);
		gtk_widget_show(value);
		processor_pipeline_entry_widget[i] = value;
	}
	gtk_widget_show(pipe_window);

	// create status frame and window
	status_frame = gtk_frame_new(NULL);
	gtk_container_set_border_width(GTK_CONTAINER(status_frame), 3);
	gtk_frame_set_shadow_type(GTK_FRAME(status_frame), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(top_vbox), status_frame, FALSE, FALSE, 0);
	gtk_widget_show(status_frame);
	status_window = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(status_frame), status_window);
	{
		char buf[256];
		GtkWidget* frame;
		GtkWidget* name;
		GtkWidget* value;
		frame = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
		gtk_box_pack_start(GTK_BOX(status_window), frame, FALSE, FALSE, 0);
		gtk_widget_show(frame);
		name = gtk_label_new("Current Status");
		gtk_misc_set_alignment(GTK_MISC(name), 0.0, 0.5);
		gtk_misc_set_padding(GTK_MISC(name), 2, 1);
		gtk_container_add(GTK_CONTAINER(frame), name);
		gtk_widget_show(name);
		value = gtk_entry_new();
		gtk_entry_set_editable(GTK_ENTRY(value), FALSE);
		gtk_box_pack_start(GTK_BOX(status_window), value, FALSE, FALSE, 0);
		gtk_widget_show(value);
		processor_status_entry_widget = value;
	}
	gtk_widget_show(status_window);
}

template <class T>
void gui_debugger<T>::create_screen_frame(void)
{
	const char buf[] = "Hello, world.";
	GtkWidget* table;
	GtkWidget* textbox;

	// create top frame
	screen_frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(screen_frame), GTK_SHADOW_NONE);
	gtk_container_border_width(GTK_CONTAINER(screen_frame), 0);
	gtk_widget_set_usize(screen_frame, 0, 100);

	// create table
	table = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(table),
								   GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(screen_frame), table);
	gtk_widget_show(table);

	// create textbox
	textbox = gtk_text_new(NULL, NULL);
	gtk_text_set_editable(GTK_TEXT(textbox), FALSE);
	gtk_container_add(GTK_CONTAINER(table), textbox);
	gtk_widget_show(textbox);
	screen_text_widget = textbox;
}

template <class T>
void gui_debugger<T>::create_disassemble_frame(void)
{
	char* titles[] = { "Address", "Code" };
	char* dummy[] = { "", "" };
	GtkWidget* table;
	GtkWidget* clist;

	// create top frame
	disassemble_frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(disassemble_frame), GTK_SHADOW_NONE);
	gtk_container_border_width(GTK_CONTAINER(disassemble_frame), 0);
	gtk_widget_set_usize(disassemble_frame, 0, 300);

	// create table
	table = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(table),
								   GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	gtk_container_add(GTK_CONTAINER(disassemble_frame), table);
	gtk_widget_show(table);

	// create clist
	clist = gtk_clist_new_with_titles(sizeof(titles) / sizeof(titles[0]),
									  titles);
	gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_IN);
	gtk_clist_set_column_justification(GTK_CLIST(clist), 0, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification(GTK_CLIST(clist), 1, GTK_JUSTIFY_LEFT);
	gtk_clist_set_column_resizeable(GTK_CLIST(clist), 0, FALSE);
	gtk_clist_set_column_resizeable(GTK_CLIST(clist), 1, FALSE);
	gtk_clist_set_column_width(GTK_CLIST(clist), 0, 64);
	gtk_clist_set_column_width(GTK_CLIST(clist), 1, 150);
	gtk_clist_set_reorderable(GTK_CLIST(clist), FALSE);
	for (size_t i = 0; i < disassemble_clist_size; i++) {
		gtk_clist_append(GTK_CLIST(clist), dummy);
	}
	gtk_container_add(GTK_CONTAINER(table), clist);
	gtk_widget_show(clist);
	disassemble_clist_widget = clist;
}

template <class T>
void gui_debugger<T>::create_memory_frame(void)
{
	char* titles[] = { "Address", "+0", "+1", "+2", "+3",
								  "+4", "+5", "+6", "+7", };
	char* dummy[] = { "", "", "", "", "", "", "", "", "", };
	GtkWidget* table;
	GtkWidget* clist;
	size_t i;

	// create top frame
	memory_frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(memory_frame), GTK_SHADOW_NONE);
	gtk_container_border_width(GTK_CONTAINER(memory_frame), 0);

	// create table
	table = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(table),
								   GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	gtk_container_add(GTK_CONTAINER(memory_frame), table);
	gtk_widget_show(table);

	// create clist
	clist = gtk_clist_new_with_titles(sizeof(titles) / sizeof(titles[0]),
									  titles);
	gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_IN);
	gtk_clist_set_column_justification(GTK_CLIST(clist), 0, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_resizeable(GTK_CLIST(clist), 0, FALSE);
	gtk_clist_set_column_width(GTK_CLIST(clist), 0, 64);
	for (i = 1; i < sizeof(titles) / sizeof(titles[0]); i++) {
		gtk_clist_set_column_justification(GTK_CLIST(clist), i,
										   GTK_JUSTIFY_CENTER);
		gtk_clist_set_column_resizeable(GTK_CLIST(clist), i, FALSE);
		gtk_clist_set_column_width(GTK_CLIST(clist), i, 16);
	}
	for (i = 0; i < memory_clist_size; i++) {
		gtk_clist_append(GTK_CLIST(clist), dummy);
	}
	gtk_container_add(GTK_CONTAINER(table), clist);
	gtk_widget_show(clist);
	memory_clist_widget = clist;
}

template <class T>
void gui_debugger<T>::create_status_frame(void)
{
	GtkWidget* hbox;
	GtkWidget* clock_frame;
	GtkWidget* clock;
	GtkWidget* empty_frame;

	// create top frame
	status_frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(status_frame), GTK_SHADOW_NONE);
	gtk_container_border_width(GTK_CONTAINER(status_frame), 0);

	// create hbox
	hbox = gtk_hbox_new(FALSE, 1);
	gtk_container_add(GTK_CONTAINER(status_frame), hbox);
	gtk_widget_show(hbox);

	// create clock frame
	clock_frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(clock_frame), GTK_SHADOW_IN);
	gtk_container_border_width(GTK_CONTAINER(clock_frame), 0);
	gtk_box_pack_start(GTK_BOX(hbox), clock_frame, FALSE, FALSE, 0);
	gtk_widget_show(clock_frame);

	// create clock label
	clock = gtk_label_new(NULL);
	gtk_widget_set_usize(clock, 200, 0);
	gtk_container_add(GTK_CONTAINER(clock_frame), clock);
	gtk_widget_show(clock);
	status_clock_label_widget = clock;

	// create empty frame
	empty_frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(empty_frame), GTK_SHADOW_IN);
	gtk_container_border_width(GTK_CONTAINER(empty_frame), 0);
	gtk_box_pack_start(GTK_BOX(hbox), empty_frame, TRUE, TRUE, 0);
	gtk_widget_show(empty_frame);
}

template <class T>
void gui_debugger<T>::create_about_window(void)
{
	GtkWidget* top_vbox;
	GtkWidget* buttonbox;
	GtkWidget* button;
	GtkWidget* notebook;
	GtkWidget* frame;
	GtkWidget* label;

	// create top window
	about_window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(about_window), "About ISIS");
	gtk_window_set_policy(GTK_WINDOW(about_window), FALSE, FALSE, FALSE);
	gtk_window_set_position(GTK_WINDOW(about_window), GTK_WIN_POS_MOUSE);
	gtk_container_border_width(GTK_CONTAINER(about_window), 5);
	gtk_signal_connect(GTK_OBJECT(about_window), "destroy",
					   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about_window);

	// create top vbox
	top_vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(about_window), top_vbox);
	gtk_widget_show(top_vbox);

	// create notebook in top vbox
	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(top_vbox), notebook, FALSE, FALSE, 0);
	gtk_widget_show(notebook);

	// create frame in notebook
	{
		GtkWidget* frame_label;
		frame = gtk_frame_new(NULL);
		gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
		frame_label = gtk_label_new("Credits");
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, frame_label);
		gtk_widget_show(frame);
	}

	// create label in frame
	label = gtk_label_new(credits_message);
	gtk_widget_set_usize(label, 400, 120);
	gtk_container_add(GTK_CONTAINER(frame), label);
	gtk_widget_show(label);

	// create button box in top vbox
	buttonbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox),
							  GTK_BUTTONBOX_DEFAULT_STYLE);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 5);
	gtk_box_pack_start(GTK_BOX(top_vbox), buttonbox, FALSE, FALSE, 0);
	gtk_widget_show(buttonbox);

	// create "Close" button in button box
	button = gtk_button_new_with_label("Close");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
							  GTK_SIGNAL_FUNC(gtk_widget_destroy),
							  GTK_OBJECT(about_window));
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(buttonbox), button, TRUE, TRUE, 0);
	gtk_widget_show(button);
	gtk_widget_grab_default(button);
}

template <class T>
void gui_debugger<T>::update_processor_frame(void)
{
	processor_type& pu(active_debugger->current_pe().processor());
	size_t i;

	// registers
	for (i = 0; i < processor_register_entry_widget.size(); i++) {
		if (pu.register_file(i) != processor_register_buf[i]) {
			processor_register_buf[i] = pu.register_file(i);
			char buf[16];
			ostrstream os(buf, sizeof(buf));
			os << hex << setfill('0') << setw(8)
			   << processor_register_buf[i] << ends;
			gtk_entry_set_text(GTK_ENTRY(processor_register_entry_widget[i]),
							   buf);
		}
	}

	// pipeline
	{
		char buf[1024];
		ostrstream os(buf, sizeof(buf));
		pu.output(os, "ibuf");
		os << ends;
		istrstream is(buf, sizeof(buf));
		for (i = 0; i < processor_pipeline_entry_widget.size(); i++) {
			char line[1024];
			is.getline(line, sizeof(line));
			if (i == 0) {
				ostrstream if_os(line, sizeof(line));
				if_os << "pc: 0x" << hex << pu.program_counter() << ends;
			}
			gtk_entry_set_text(GTK_ENTRY(processor_pipeline_entry_widget[i]),
							   line);
		}
	}

	// status
	{
		char buf[1024];
		ostrstream os(buf, sizeof(buf));
		pu.output(os, "stall");
		os << ends;
		string str(buf);
		if (str == "empty") str = "running";
		gtk_entry_set_text(GTK_ENTRY(processor_status_entry_widget),
									 str.c_str());
	}
}

template <class T>
void gui_debugger<T>::update_screen_frame(void)
{
}

template <class T>
void gui_debugger<T>::update_disassemble_frame(void)
{
	processing_element_type& pe(active_debugger->current_pe());
	address_type adr = disassemble_clist_address;
	for (size_t i = 0; i < disassemble_clist_size; i++) {
		char buf[256];
		if (pe.memory_map().is_valid(adr)) {
			{
				ostrstream os(buf, sizeof(buf));
				os << hex << setfill('0') << setw(8) << adr << ends;
				gtk_clist_set_text(GTK_CLIST(disassemble_clist_widget), i, 0,
								   buf);
			}
			{
				ostrstream os(buf, sizeof(buf));
				pe.disassemble(os, adr, sizeof_data_type);
				os << ends;
				string str1(buf);
				size_t idx;
				idx = str1.find_first_of(' ');
				idx = str1.find_first_of(' ', idx + 1);
				string str2(str1.substr(idx + 2));
				gtk_clist_set_text(GTK_CLIST(disassemble_clist_widget), i, 1,
								   str2.c_str());
			}
		} else {
			gtk_clist_set_text(GTK_CLIST(disassemble_clist_widget), i, 0,
							   "--------");
			gtk_clist_set_text(GTK_CLIST(disassemble_clist_widget), i, 1, "-");
		}
		adr += sizeof_data_type;
	}
}

template <class T>
void gui_debugger<T>::update_memory_frame(void)
{
	processing_element_type& pe(active_debugger->current_pe());
	address_type adr = memory_clist_address;
	for (size_t i = 0; i < memory_clist_size; i++) {
		char buf[16];
		if (pe.memory_map().is_valid(adr)) {
			{
				ostrstream os(buf, sizeof(buf));
				os << hex << setfill('0') << setw(8) << adr << ends;
				gtk_clist_set_text(GTK_CLIST(memory_clist_widget), i, 0, buf);
			}
			for (size_t j = 0; j < 8; j++) {
				unsigned char data
					= (unsigned char)(pe.memory_map().read_char(adr + j));
				ostrstream os(buf, sizeof(buf));
				os << hex << setfill('0') << setw(2)
				   << (unsigned int)(data) << ends;
				gtk_clist_set_text(GTK_CLIST(memory_clist_widget), i, j + 1,
								   buf);
			}
		} else {
			gtk_clist_set_text(GTK_CLIST(memory_clist_widget), i, 0,
							   "--------");
			for (size_t j = 0; j < 8; j++) {
				gtk_clist_set_text(GTK_CLIST(memory_clist_widget), i, j, "--");
			}
		}
		adr += 8;
	}
}

template <class T>
void gui_debugger<T>::update_status_frame(void)
{
	processing_element_type& pe(active_debugger->current_pe());
	char buf[1024];
	ostrstream os(buf, sizeof(buf));
	os << "current clock: " << pe.timer_clock_value() << ends;
	gtk_label_set_text(GTK_LABEL(status_clock_label_widget), buf);
}

template <class T>
void gui_debugger<T>::show_about_window(void)
{
	if (about_window == NULL) create_about_window();
	gtk_widget_show(about_window);
}

template <class T>
void gui_debugger<T>::step_callback(void)
{
	thisclass* me = active_debugger;
	if (me->is_halt() || me->is_bus_error()) return;
	me->step();
	update_processor_frame();
	update_screen_frame();
	update_status_frame();
}

template <class T>
void gui_debugger<T>::continue_callback(void)
{
	thisclass* me = active_debugger;
	if (me->is_halt() || me->is_bus_error()) return;
	processor_type& pu(active_debugger->current_pe().processor());
	vector<address_type> breakpoints;
	me->dump_breakpoints(breakpoints);
	address_type pc = pu.program_counter();
	size_t count = 0;
	while (!me->is_halt() && !me->is_bus_error()) {
		me->step();
		address_type new_pc = pu.program_counter();
		if (new_pc != pc) {
			pc = new_pc;
			size_t i;
			for (i = 0; i < breakpoints.size(); i++) {
				if (breakpoints[i] == pc) break;
			}
			if (i < breakpoints.size()) break;
		}
		if (++count == 16384) {
			update_processor_frame();
			update_screen_frame();
			update_status_frame();
			count = 0;
		}
	}
	update_processor_frame();
	update_screen_frame();
	update_status_frame();
}

#else // !USE_GUI

#include "debugger.h"

template <class T>
class gui_debugger : public debugger<T>
{
private:
	typedef gui_debugger<T> thisclass;
	typedef debugger<T> inherited;
public:
	gui_debugger(void);
	gui_debugger(const gui_debugger<T>&);
	virtual ~gui_debugger();
	void interactive_mode(void);
};

template <class T>
gui_debugger<T>::gui_debugger(void)
	: inherited()
{}

template <class T>
gui_debugger<T>::gui_debugger(const gui_debugger<T>& a)
	: inherited(a)
{}

template <class T>
gui_debugger<T>::~gui_debugger()
{}

template <class T>
void gui_debugger<T>::interactive_mode(void)
{
	cerr << "Sorry, you cannot use GUI debugger." << endl
		 << "You must re-install ISIS with configure option '--enable-gui'."
		 << endl;
}

#endif // USE_GUI

#endif /* GUI_DEBUGGER_H */
