/* -*- C++ -*-
 *
 * <<< random_access_processor_base.h >>>
 *
 * --- Abstract random-access processor class
 *     'random_access_processor_base'
 *     Copyright (C) 1997-1999 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef RANDOM_ACCESS_PROCESSOR_BASE_H
#define RANDOM_ACCESS_PROCESSOR_BASE_H 1

#include <iostream>
#include "processor.h"
#include "bus_port.h"

template <class A, class D = A>
class random_access_processor_base : public processor<A, D>
{
private:
	typedef random_access_processor_base<A, D> thisclass;
	typedef processor<A, D> inherited;
public:
	typedef typename inherited::address_type address_type;
	typedef typename inherited::data_type data_type;
	typedef bus_port<address_type, data_type> port_type;
private:
	enum state_type {
		READY,
		READ_REQ_OWNER,
		READ_SEND,
		READ_WAIT,
		READ_GRANT,
		READ_ACK,
		READ_NACK,
		READ_TRANS,
		WRITE_REQ_OWNER,
		WRITE_SEND,
		WRITE_WAIT,
		WRITE_GRANT,
		WRITE_ACK,
		WRITE_NACK,
		WRITE_TRANS,
		HALT
	};
	port_type bus_if;
	state_type state;
	double access_probability_, read_probability_;
	bool read_wait_flag, write_wait_flag;
	address_type req_adr;
	typename port_type::transaction_id_type trans_id;
public:
	inline static unsigned long unsigned_long_random(void);
	inline static double double_random(void);
	random_access_processor_base(void);
	random_access_processor_base(const thisclass&);
	virtual ~random_access_processor_base();
	virtual void output(ostream&) const;
	virtual const data_type& register_file(size_t) const
		{ static data_type dummy = 0; return dummy; }
	virtual data_type& register_file(size_t)
		{ static data_type dummy = 0; return dummy; }
	virtual const address_type& program_counter(void) const
		{ static address_type dummy = 0; return dummy; }
	virtual address_type& program_counter(void)
		{ static address_type dummy = 0; return dummy; }
	virtual bool is_halt(void) const { return state == HALT; }
	inline virtual bool is_reading(void) const;
	inline virtual bool is_writing(void) const;
	virtual bool is_stall(void) const { return is_reading(); }
	virtual bool is_user_mode(void) const { return true; }
	virtual bool is_kernel_mode(void) const { return false; }
	virtual void clock_in(void);
	virtual void clock_out(void);
	virtual void reset(void);
	virtual void halt(void);
	virtual const port& port_ref(void) const { return bus_if; }
	virtual port& port_ref(void) { return bus_if; }
	double access_probability(void) const { return access_probability_; }
	double read_probability(void) const { return read_probability_; }
	double write_probability(void) const { return 1 - read_probability_; }
	void set_access_probability(double);
	void set_read_probability(double);
	void set_write_probability(double);
	virtual address_type get_read_address(void) const = 0;
	virtual address_type get_write_address(void) const = 0;
	virtual data_type get_write_data(void) const = 0;
};

template <class A, class D>
inline unsigned long random_access_processor_base<A, D>::unsigned_long_random
	(void)
{
	static unsigned long seed;
	return seed = 1566083941UL * seed + 1;
}

template <class A, class D>
inline double random_access_processor_base<A, D>::double_random(void)
{
	return double(unsigned_long_random() >> 16) / 0x10000UL;
}

template <class A, class D>
bool random_access_processor_base<A, D>::is_reading(void) const
{
	switch (state) {
	case READ_REQ_OWNER:
	case READ_SEND:
	case READ_WAIT:
	case READ_ACK:
	case READ_NACK:
	case READ_TRANS:
		return true;
	default:
		break;
	}
	return false;
}

template <class A, class D>
bool random_access_processor_base<A, D>::is_writing(void) const
{
	switch (state) {
	case WRITE_REQ_OWNER:
	case WRITE_SEND:
	case WRITE_WAIT:
	case WRITE_ACK:
	case WRITE_NACK:
	case WRITE_TRANS:
		return true;
	default:
		break;
	}
	return false;
}

template <class A, class D>
random_access_processor_base<A, D>::random_access_processor_base(void)
	: state(READY),
	  access_probability_(0),
	  read_probability_(1),
	  read_wait_flag(false),
	  write_wait_flag(false)
{}

template <class A, class D>
random_access_processor_base<A, D>::random_access_processor_base(
	const random_access_processor_base<A, D>& a)
	: state(a.state),
	  access_probability_(a.access_probability_),
	  read_probability_(a.read_probability_),
	  read_wait_flag(a.read_wait_flag),
	  write_wait_flag(a.write_wait_flag)
{}

template <class A, class D>
random_access_processor_base<A, D>::~random_access_processor_base()
{}

template <class A, class D>
void random_access_processor_base<A, D>::clock_in(void)
{
	if (!read_wait_flag && !write_wait_flag &&
		double_random() < access_probability()) {
		if (double_random() < read_probability()) {
			read_wait_flag = true;
		} else {
			write_wait_flag = true;
		}
	}
	switch (state) {
	case READY:
		if (bus_if.is_owned()) break;
		if (write_wait_flag) {
			req_adr = get_write_address();
			state = WRITE_REQ_OWNER;
		} else if (read_wait_flag) {
			req_adr = get_read_address();
			state = READ_REQ_OWNER;
		}
		break;
	case READ_REQ_OWNER:
		if (bus_if.is_owner()) {
			trans_id = bus_if.transaction_id();
			state = READ_SEND;
		}
		break;
	case READ_WAIT:
		if (bus_if.transaction_id() == trans_id) {
			if (bus_if.is_single_read_grant()) {
				state = READ_GRANT;
			} else if (bus_if.is_single_read_ack()) {
				state = READ_ACK;
			} else if (bus_if.is_single_read_nack()) {
				state = READ_NACK;
			}
		}
		break;
	case READ_ACK:
		if (bus_if.is_single_read_data()) {
			state = READ_TRANS;
		}
		break;
	case READ_NACK:
		state = READ_REQ_OWNER;
		break;
	case WRITE_REQ_OWNER:
		if (bus_if.is_owner()) {
			trans_id = bus_if.transaction_id();
			state = WRITE_SEND;
		}
		break;
	case WRITE_WAIT:
		if (bus_if.transaction_id() == trans_id) {
			if (bus_if.is_single_write_grant()) {
				state = WRITE_GRANT;
			} else if (bus_if.is_single_write_ack()) {
				state = WRITE_ACK;
			} else if (bus_if.is_single_write_nack()) {
				state = WRITE_NACK;
			}
		}
		break;
	case WRITE_NACK:
		state = WRITE_REQ_OWNER;
		break;
	default:
		break;
	}
}

template <class A, class D>
void random_access_processor_base<A, D>::clock_out(void)
{
	switch (state) {
	case READ_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case READ_SEND:
		bus_if.send_single_read_request(req_adr);
		state = READ_WAIT;
		break;
	case READ_GRANT:
		bus_if.clear();
		bus_if.release_ownership();
		state = READ_WAIT;
		break;
	case READ_NACK:
		bus_if.clear();
		bus_if.release_ownership();
		break;
	case READ_TRANS:
		bus_if.clear();
		bus_if.release_ownership();
		read_wait_flag = false;
		state = READY;
		break;
	case WRITE_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case WRITE_SEND:
		bus_if.send_single_write_request(req_adr);
		state = WRITE_WAIT;
		break;
	case WRITE_GRANT:
		bus_if.clear();
		bus_if.release_ownership();
		state = WRITE_WAIT;
		break;
	case WRITE_ACK:
		bus_if.send_single_write_data(get_write_data());
		state = WRITE_TRANS;
		break;
	case WRITE_NACK:
		bus_if.clear();
		bus_if.release_ownership();
		break;
	case WRITE_TRANS:
		bus_if.clear();
		bus_if.release_ownership();
		write_wait_flag = false;
		state = READY;
		break;
	default:
		break;
	}
}

template <class A, class D>
void random_access_processor_base<A, D>::reset(void)
{
	state = READY;
	read_wait_flag = write_wait_flag = false;
}

template <class A, class D>
void random_access_processor_base<A, D>::halt(void)
{
	state = HALT;
}

template <class A, class D>
void random_access_processor_base<A, D>::set_access_probability(double a)
{
	if (a >= 0 && a <= 1) access_probability_ = a;
}

template <class A, class D>
void random_access_processor_base<A, D>::set_read_probability(double a)
{
	if (a >= 0 && a <= 1) read_probability_ = a;
}

template <class A, class D>
void random_access_processor_base<A, D>::set_write_probability(double a)
{
	set_read_probability(1 - a);
}

template <class A, class D>
void random_access_processor_base<A, D>::output(ostream& os) const
{
	const long flags = os.flags();
	os << hex;
	switch (state) {
	case READY:
		os << "ready";
		break;
	case READ_REQ_OWNER:
		os << "read_request_ownership(0x" << req_adr << ')';
		break;
	case READ_SEND:
		os << "read_send(0x" << req_adr << ')';
		break;
	case READ_WAIT:
		os << "read_wait(0x" << req_adr << ')';
		break;
	case READ_GRANT:
		os << "read_grant(0x" << req_adr << ')';
		break;
	case READ_ACK:
		os << "read_ack(0x" << req_adr << ')';
		break;
	case READ_NACK:
		os << "read_nack(0x" << req_adr << ')';
		break;
	case READ_TRANS:
		os << "read_trans(0x" << req_adr << ')';
		break;
	case WRITE_REQ_OWNER:
		os << "write_request_ownership(0x" << req_adr << ')';
		break;
	case WRITE_SEND:
		os << "write_send(0x" << req_adr << ')';
		break;
	case WRITE_WAIT:
		os << "write_wait(0x" << req_adr << ')';
		break;
	case WRITE_GRANT:
		os << "write_grant(0x" << req_adr << ')';
		break;
	case WRITE_ACK:
		os << "write_ack(0x" << req_adr << ')';
		break;
	case WRITE_NACK:
		os << "write_nack(0x" << req_adr << ')';
		break;
	case WRITE_TRANS:
		os << "write_trans(0x" << req_adr << ')';
		break;
	case HALT:
		os << "halt";
		break;
	}
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* RANDOM_ACCESS_PROCESSOR_BASE_H */
