/* -*- C++ -*-
 *
 * <<< halt_detector.h >>>
 *
 * --- Halt detector unit class 'halt_detector'
 *     Copyright (C) 1997-1999 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef HALT_DETECTOR_H
#define HALT_DETECTOR_H 1

#include <iostream>
#include "single_port_synchronous_unit.h"
#include "bus_port.h"

template <class A, class D = A>
class halt_detector : public single_port_synchronous_unit
{
private:
	typedef halt_detector<A, D> thisclass;
	typedef single_port_synchronous_unit inherited;
public:
	typedef bus_port<A, D> port_type;
	typedef typename port_type::address_type address_type;
	typedef typename port_type::data_type data_type;
private:
	enum state_t { READY, ACK } state;
	port_type bus_if;
	address_type adr;
	bool halt_flag;
public:
	halt_detector(void);
	halt_detector(const address_type&);
	halt_detector(const thisclass&);
	virtual ~halt_detector();
	virtual void output(ostream&) const;
	bool is_halt(void) const { return halt_flag; }
	const address_type& address(void) const { return adr; }
	void set_address(const address_type& a) { adr = a; }
	virtual void clock_in(void);
	virtual void clock_out(void);
	virtual void reset(void) { halt_flag = false; }
	virtual const port& port_ref(void) const { return bus_if; }
	virtual port& port_ref(void) { return bus_if; }
};

template <class A, class D>
halt_detector<A, D>::halt_detector(void)
	: state(READY), adr(address_type(0)), halt_flag(false)
{}

template <class A, class D>
halt_detector<A, D>::halt_detector(const halt_detector<A, D>::address_type& a)
	: state(READY), adr(a), halt_flag(false)
{}

template <class A, class D>
halt_detector<A, D>::halt_detector(const halt_detector<A, D>& a)
	: state(a.state), adr(a.adr), halt_flag(a.halt_flag)
{}

template <class A, class D>
halt_detector<A, D>::~halt_detector()
{}

template <class A, class D>
void halt_detector<A, D>::clock_in(void)
{
	if (bus_if.is_single_write_request() && bus_if.address() == address()) {
		halt_flag = true;
		state = ACK;
	}
}

template <class A, class D>
void halt_detector<A, D>::clock_out(void)
{
	if (state == ACK) {
		bus_if.send_single_write_ack();
		state = READY;
	}
}

template <class A, class D>
void halt_detector<A, D>::output(ostream& os) const
{
	os << (is_halt() ? "halt" : "not halt");
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* HALT_DETECTOR_H */
