/*
 * <<< r3000_processing_element.cc >>>
 *
 * --- R3000 processing element class 'r3000_processing_element'
 *     Copyright (C) 1997-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <iostream>
#include "r3000_processing_element.h"
#include "mips_executable.h"

const r3000_processing_element::address_type
	r3000_processing_element::default_multi_io_area_address = 0xbfd00000;
const r3000_processing_element::size_type
	r3000_processing_element::default_multi_io_area_size = 0x100;

r3000_processing_element::r3000_processing_element(void)
{
	set_multi_io_area(default_multi_io_area_address,
		default_multi_io_area_size);
}

r3000_processing_element::r3000_processing_element
	(const r3000_processing_element& a)
	: inherited(a),
	  inst_cache(a.inst_cache),
	  data_cache(a.data_cache),
	  fpa(a.fpa)
{
	pu.connect_instruction_cache(inst_cache);
	pu.connect_data_cache(data_cache);
	if (a.is_connected_to_coprocessor()) pu.connect_coprocessor(fpa);
}

r3000_processing_element::~r3000_processing_element()
{}

void r3000_processing_element::output(ostream& os) const
{
	os << pu;
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

bool r3000_processing_element::output(ostream& os, const string& s) const
{
	if (s == "pu") {
		os << pu;
		return true;
	} else if (s == "fpa" || s == "cp1") {
		os << fpa;
		return true;
	} else if (s == "instruction_cache" || s == "icache") {
		os << inst_cache;
		return true;
	} else if (s == "data_cache" || s == "dcache") {
		os << data_cache;
		return true;
	}
	return pu.output(os, s);
}

bool r3000_processing_element::load(const char* a)
{
	ifstream file(a, ios::in | ios::nocreate);
	if (file.bad()) return false;
	mips_executable object_file;
	object_file.load_header(file);
	if (!object_file.is_valid()) return false;
	bool flag = object_file.load_body_to_memory(file, memory_map());
	if (flag) {
		pu.program_counter() = object_file.entry_point();
	}
	return flag;
}

void r3000_processing_element::disassemble
	(ostream& os,
	 const r3000_processing_element::address_type& top,
	 const r3000_processing_element::size_type& size) const
{
	mips_executable::disassemble(os, memory_map(), top, size);
}

void r3000_processing_element::set_instruction_cache_size
	(r3000_processing_element::size_type a,
	 r3000_processing_element::size_type b)
{
	if (a > 0 && b > 0) {
		inst_cache.resize(a, b);
		pu.connect_instruction_cache(inst_cache);
	} else {
		inst_cache.resize(0, 0);
		pu.disconnect_instruction_cache();
	}
}

void r3000_processing_element::set_data_cache_size
	(r3000_processing_element::size_type a,
	 r3000_processing_element::size_type b)
{
	if (a > 0 && b > 0) {
		data_cache.resize(a, b);
		pu.connect_data_cache(data_cache);
	} else {
		data_cache.resize(0, 0);
		pu.disconnect_data_cache();
	}
}

void r3000_processing_element::connect_coprocessor(void)
{
	pu.connect_coprocessor(fpa);
}

void r3000_processing_element::disconnect_coprocessor(void)
{
	pu.disconnect_coprocessor();
}
