/* -*- C++ -*-
 *
 * <<< r3000_pipeline_stage.h >>>
 *
 * --- R3000 pipeline state class 'r3000_pipeline_stage'
 *     Copyright (C) 1995-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef R3000_PIPELINE_STAGE_H
#define R3000_PIPELINE_STAGE_H 1

#include <string>
#include "r3000_typedef.h"
#include "r3000_instruction.h"
#include "r3000_register_file.h"

class ostream;

class r3000_pipeline_stage
{
private:
	typedef r3000_pipeline_stage thisclass;
	struct register_set
	{
		r3000_word value;
		int number;
		inline register_set(void);
		inline register_set(r3000_word, int);
		inline void clear(void);
	};
	enum io_flag_value {
		int_in	= 1,
		int_out = 2,
		cp_in	= 4,
		cp_out	= 8,
		dst_out = 16
	};
	r3000_instruction buf;
	register_set src1, src2, dst;
	int reg_in, io_flags;
	r3000_word acc_adr, acc_dat, dst_pc;
	bool is_destination_output(void) const { return io_flags & dst_out; }
public:
	r3000_pipeline_stage(void);
	r3000_pipeline_stage(const thisclass&);
	~r3000_pipeline_stage();
	r3000_instruction::operation_type operation(void) const
		{ return buf.operation(); }
	r3000_instruction::opcode_type opcode(void) const
		{ return buf.opcode(); }
	bool is_integer_input(void) const { return io_flags & int_in; }
	bool is_integer_output(void) const { return io_flags & int_out; }
	bool is_coprocessor_input(void) const { return io_flags & cp_in; }
	bool is_coprocessor_output(void) const { return io_flags & cp_out; }
	int coprocessor_number(void) const { return buf.coprocessor_number(); }
	r3000_word image(void) const { return buf.image(); }
	int source1_number(void) const { return src1.number; }
	int source2_number(void) const { return src2.number; }
	int destination_number(void) const { return dst.number; }
	r3000_word source1(void) const { return src1.value; }
	r3000_word source2(void) const { return src2.value; }
	r3000_word destination(void) const { return dst.value; }
	r3000_word access_address(void) const { return acc_adr; }
	r3000_word word_aligned_access_address(void) const
		{ return access_address() & ~r3000_word(sizeof_r3000_word - 1); }
	r3000_word access_data(void) const { return acc_dat; }
	r3000_word destination_pc(void) const { return dst_pc; }
	bool is_load(void) const { return buf.is_load(); }
	bool is_store(void) const { return buf.is_store(); }
	bool is_load_or_store(void) const { return buf.is_load_or_store(); }
	r3000_word break_code(void) const { return buf.break_code(); }
	string opcode_string(void) const { return buf.opcode_string(); }
	string operation_string(void) const { return buf.operation_string(); }
	void instruction_fetch(r3000_word, r3000_word);
	inline void register_fetch(const r3000_register_file&);
	inline void coprocessor_condition_fetch(int);
	inline void forwarding(const r3000_pipeline_stage&);
	void execute(void);
	void load_fetch_for_big_endian(r3000_word);
	void load_fetch_for_little_endian(r3000_word);
	void partial_word_store_fetch_for_big_endian(r3000_word);
	void partial_word_store_fetch_for_little_endian(r3000_word);
	inline void mf_fetch(r3000_word);
	inline void coprocessor_store_fetch(r3000_word);
	inline void writeback(r3000_register_file&);
	void clear(void);
	void output(ostream& os) const;
};

ostream& operator<<(ostream&, const r3000_pipeline_stage&);

inline void r3000_pipeline_stage::register_fetch(const r3000_register_file& rf)
{
	switch (reg_in) {
	case 1:
		src1.value = rf[source1_number()];
		break;
	case 2:
		src1.value = rf[source1_number()];
		src2.value = rf[source2_number()];
		break;
	default:
		break;
	}
}

inline void r3000_pipeline_stage::coprocessor_condition_fetch(int a)
{
	src1.value = a;
}

inline void r3000_pipeline_stage::forwarding(const r3000_pipeline_stage& st)
{
	if (!st.is_destination_output()) return;
	const int dst_number = st.destination_number();
	switch (reg_in) {
	case 1:
		if (source1_number() == dst_number) src1.value = st.destination();
		break;
	case 2:
		if (source1_number() == dst_number) src1.value = st.destination();
		if (source2_number() == dst_number) src2.value = st.destination();
		break;
	}
}

inline void r3000_pipeline_stage::mf_fetch(r3000_word a)
{
	dst.value = a;
}

inline void r3000_pipeline_stage::coprocessor_store_fetch(r3000_word a)
{
	acc_dat = a;
}

inline void r3000_pipeline_stage::writeback(r3000_register_file& rf)
{
	if (is_destination_output()) rf[destination_number()] = destination();
}

#endif /* R3000_PIPELINE_STAGE_H */
