/* -*- C++ -*-
 *
 * <<< r3000_multiply_unit.h >>>
 *
 * --- R3000 multiply/divide unit class 'r3000_multiply_unit'
 *     Copyright (C) 1995-1999 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef R3000_MULTIPLY_UNIT_H
#define R3000_MULTIPLY_UNIT_H 1

#include "r3000_typedef.h"

class r3000_multiply_unit
{
private:
	typedef r3000_multiply_unit thisclass;
	static const unsigned int default_mult_wait, default_div_wait;
	r3000_word hi_, lo_;
	unsigned int mult_wait_, div_wait_, count;
public:
	r3000_multiply_unit(void);
	r3000_multiply_unit(const thisclass&);
	bool ready(void) const { return count == 0; }
	bool busy(void) const { return count > 0; }
	const r3000_word& hi(void) const { return hi_; }
	const r3000_word& lo(void) const { return lo_; }
	r3000_word& hi(void) { return hi_; }
	r3000_word& lo(void) { return lo_; }
	inline void mult(r3000_word, r3000_word);
	inline void multu(r3000_word, r3000_word);
	inline void div(r3000_word, r3000_word);
	inline void divu(r3000_word, r3000_word);
	void clock(void) { if (busy()) count--; }
	void clear(void);
	unsigned int mult_wait(void) const { return mult_wait_; }
	unsigned int div_wait(void) const { return div_wait_; }
	void set_mult_wait(unsigned int);
	void set_div_wait(unsigned int);
};

inline void r3000_multiply_unit::multu(r3000_word a, r3000_word b)
{
	r3000_word a_hi = (a >> 16), a_lo = a & 0xffffU,
			   b_hi = (b >> 16), b_lo = b & 0xffffU;
	r3000_word tmp1 = a_lo * b_lo;
	r3000_word tmp2 = (a_hi * b_lo + a_lo * b_hi + (tmp1 >> 16));
	lo_ = (tmp2 << 16) | (tmp1 & 0xffffU);
	hi_ = a_hi * b_hi + (tmp2 >> 16);
	count = mult_wait();
}

inline void r3000_multiply_unit::mult(r3000_word a, r3000_word b)
{
	r3000_word a_ = ((a & 0x80000000UL) == 0) ? a : (~a + 1);
	r3000_word b_ = ((b & 0x80000000UL) == 0) ? b : (~b + 1);
	multu(a_, b_);
	if ((a ^ b) & 0x80000000UL) {
		if (lo_ != 0) {
			lo_ = ~lo_ + 1;
			hi_ = ~hi_;
		} else {
			hi_ = ~hi_ + 1;
		}
	}
}

inline void r3000_multiply_unit::div(r3000_word a, r3000_word b)
{
	signed_r3000_word a_ = signed_r3000_word(a), b_ = signed_r3000_word(b);
	lo_ = r3000_word(a_ / b_);
	hi_ = r3000_word(a_ % b_);
	count = div_wait();
}

inline void r3000_multiply_unit::divu(r3000_word a, r3000_word b)
{
	lo_ = a / b;
	hi_ = a % b;
	count = div_wait();
}

#endif /* R3000_MULTIPLY_UNIT_H */
