/*
 * <<< mips_executable.cc >>>
 *
 * --- Executable file for MIPS 'mips_executable'
 *     Copyright (C) 1995-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstddef>
#include <iostream>
#include "mips_executable.h"

istream& operator>>(istream&, mips_executable::header::file_header&);
istream& operator>>(istream&, mips_executable::header::aout_header&);
istream& operator>>(istream&, mips_executable::header::section_header&);
istream& operator>>(istream&, mips_executable::header&);

const unsigned short MIPSEBMAGIC = 0x0160;

template <class T>
static void get_big_endian(istream& is, size_t size, T& value)
{
	value = 0;
	for (size_t i = 0; i < size; i++) {
		value = (value << 8) + T(is.get());
	}
}

bool mips_executable::header::file_header::is_mips_big_endian_object(void) const
{
	return f_magic == MIPSEBMAGIC;
}

istream& operator>>(istream& is, mips_executable::header::file_header& a)
{
	get_big_endian(is, 2, a.f_magic);
	get_big_endian(is, 2, a.f_nscns);
	get_big_endian(is, 4, a.f_timdat);
	get_big_endian(is, 4, a.f_symptr);
	get_big_endian(is, 4, a.f_nsyms);
	get_big_endian(is, 2, a.f_opthdr);
	get_big_endian(is, 2, a.f_flags);
	return is;
}

istream& operator>>(istream& is, mips_executable::header::aout_header& a)
{
	get_big_endian(is, 2, a.magic);
	get_big_endian(is, 2, a.vstamp);
	get_big_endian(is, 4, a.tsize);
	get_big_endian(is, 4, a.dsize);
	get_big_endian(is, 4, a.bsize);
	get_big_endian(is, 4, a.entry);
	get_big_endian(is, 4, a.text_start);
	get_big_endian(is, 4, a.data_start);
	get_big_endian(is, 4, a.bss_start);
	get_big_endian(is, 4, a.gprmask);
	get_big_endian(is, 4, a.cprmask[0]);
	get_big_endian(is, 4, a.cprmask[1]);
	get_big_endian(is, 4, a.cprmask[2]);
	get_big_endian(is, 4, a.cprmask[3]);
	get_big_endian(is, 4, a.gp_value);
	return is;
}

istream& operator>>(istream& is, mips_executable::header::section_header& a)
{
	for (size_t i = 0; i < sizeof(a.s_name); i++) a.s_name[i] = is.get();
	get_big_endian(is, 4, a.s_paddr);
	get_big_endian(is, 4, a.s_vaddr);
	get_big_endian(is, 4, a.s_size);
	get_big_endian(is, 4, a.s_scnptr);
	get_big_endian(is, 4, a.s_relptr);
	get_big_endian(is, 4, a.s_lnnoptr);
	get_big_endian(is, 2, a.s_nreloc);
	get_big_endian(is, 2, a.s_nlnno);
	get_big_endian(is, 4, a.s_flags);
	return is;
}

istream& operator>>(istream& is, mips_executable::header& a)
{
	return is >> a.ex_f >> a.ex_o;
}

mips_executable::mips_executable(void)
	: section_header_(NULL)
{}

mips_executable::~mips_executable()
{
	if (valid_flag) delete[] section_header_;
}

void mips_executable::load_header(istream& is)
{
	if (valid_flag) delete[] section_header_;
	is >> header_;
	const size_t section_number = header_.section_number();
	if (section_number == 0) {
		valid_flag = false;
	} else {
		long first_section_ptr = 0;
		section_header_ = new header::section_header[section_number];
		for (size_t i = 0; i < section_number; i++) {
			is >> section_header_[i];
			long ptr = section_header_[i].data_pointer();
			if (ptr != 0 && (i == 0 || ptr < first_section_ptr)) {
				first_section_ptr = ptr;
			}
		}
		long loaded_ptr = sizeof_file_header + sizeof_aout_header
						+ sizeof_section_header * section_number;
		if (loaded_ptr < first_section_ptr) {
			is.ignore(first_section_ptr - loaded_ptr);
		}
		valid_flag = true;
	}
}
