/*
 * <<< multi_io_funcs.c >>>
 *
 * --- Communication routines with isis::multi_io_unit 'multi_io_funcs'
 *     Copyright (C) 2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include "multi_io_funcs.h"
#include "comm_io_funcs.h"
#include "io_protocol.h"

/* Be careful! This define is for only big-endian host. */
#define WORDS_BIGENDIAN

#ifdef WORDS_BIGENDIAN

#define binary_to_int(a) (*(int*)(a))
#define binary_to_size_t(a) (*(size_t*)(a))
#define int_to_binary(a, b) { *(int*)(b) = a; }
#define size_t_to_binary(a, b) { *(size_t*)(b) = a; }

#else /* WORDS_BIGENDIAN */

static int binary_to_int(const char*);
static size_t binary_to_size_t(const char*);
static void int_to_binary(int, char*);
static void size_t_to_binary(size_t, char*);

static int binary_to_int(const char* a)
{
	return ((int)a[0] << 24) | ((int)a[1] << 16) |
		   ((int)a[2] <<  8) | (int)a[3];
}

static size_t binary_to_size_t(const char* a)
{
	return ((size_t)a[0] << 24) | ((size_t)a[1] << 16) |
		   ((size_t)a[2] <<  8) | (size_t)a[3];
}

static void int_to_binary(int a, char* ret)
{
	ret[0] = ((a >> 24) & 0xff);
	ret[1] = ((a >> 16) & 0xff);
	ret[2] = ((a >>  8) & 0xff);
	ret[3] = (a & 0xff);
}

static void size_t_to_binary(size_t a, char* ret)
{
	ret[0] = ((a >> 24) & 0xff);
	ret[1] = ((a >> 16) & 0xff);
	ret[2] = ((a >>  8) & 0xff);
	ret[3] = (a & 0xff);
}

#endif /* WORDS_BIGENDIAN */

void __multi_io_halt(void)
{
	char c = (char)IO_HALT_REQUEST;
	__comm_io_send_to_host(&c, 1);
	__comm_io_receive_from_host(&c, 1);
}

int __multi_io_sysinfo_read_integer(const char* name, int* ret)
{
	char comm_buf[sizeof(int)];
	const size_t len = (size_t)strlen(name);
	comm_buf[0] = (char)IO_SYSINFO_READ_INTEGER_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	size_t_to_binary(len, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(len));
	__comm_io_send_to_host(name, len);
	__comm_io_receive_from_host(comm_buf, 1);
	if (comm_buf[0] == IO_COMMON_ACK) {
		__comm_io_receive_from_host(comm_buf, sizeof(*ret));
		*ret = binary_to_int(comm_buf);
		return 1;
	} else {
		return 0;
	}
}

int __multi_io_sysinfo_read_string(const char* name, char* ret, size_t ret_max)
{
	char comm_buf[sizeof(int)];
	const size_t len = (size_t)strlen(name);
	comm_buf[0] = (char)IO_SYSINFO_READ_STRING_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	size_t_to_binary(len, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(len));
	__comm_io_send_to_host(name, len);
	__comm_io_receive_from_host(comm_buf, 1);
	if (comm_buf[0] == IO_COMMON_ACK) {
		size_t ret_len;
		__comm_io_receive_from_host(comm_buf, sizeof(ret_len));
		ret_len = binary_to_size_t(comm_buf);
		if (ret_len < ret_max - 1) {
			__comm_io_receive_from_host(ret, ret_len);
			ret[ret_len] = '\0';
			return 1;
		} else {
			while (ret_len > 0) {
				size_t t = (ret_len < ret_max) ? ret_len : ret_max;
				__comm_io_receive_from_host(ret, t);
				ret_len -= t;
			}
			return 0;
		}
	} else {
		return 0;
	}
}

int __multi_io_sysinfo_write_integer(const char* name, int value)
{
	char comm_buf[sizeof(int)];
	const size_t len = (size_t)strlen(name);
	comm_buf[0] = (char)IO_SYSINFO_WRITE_INTEGER_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	size_t_to_binary(len, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(len));
	__comm_io_send_to_host(name, len);
	int_to_binary(value, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(value));
	__comm_io_receive_from_host(comm_buf, 1);
	return (comm_buf[0] == IO_COMMON_ACK) ? 1 : 0;
}

int __multi_io_sysinfo_write_string(const char* name, const char* value)
{
	char comm_buf[sizeof(int)];
	const size_t name_len = (size_t)strlen(name),
				 value_len = (size_t)strlen(value);
	comm_buf[0] = (char)IO_SYSINFO_WRITE_STRING_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	size_t_to_binary(name_len, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(name_len));
	__comm_io_send_to_host(name, name_len);
	size_t_to_binary(value_len, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(value_len));
	__comm_io_send_to_host(value, value_len);
	__comm_io_receive_from_host(comm_buf, 1);
	return (comm_buf[0] == IO_COMMON_ACK) ? 1 : 0;
}

int __multi_io_file_open(const char *name, int flags)
{
	char comm_buf[sizeof(int)];
	char fileio_flags = 0;
	size_t len = (size_t)strlen(name);
	if ((flags & O_RDONLY) == O_RDONLY) fileio_flags |= IO_FILE_OPEN_RDONLY;
	if ((flags & O_WRONLY) == O_WRONLY) fileio_flags |= IO_FILE_OPEN_WRONLY;
	if ((flags & O_RDWR  ) == O_RDWR  ) fileio_flags |= IO_FILE_OPEN_RDWR;
	if ((flags & O_APPEND) == O_APPEND) fileio_flags |= IO_FILE_OPEN_APPEND;
	if ((flags & O_CREAT ) == O_CREAT ) fileio_flags |= IO_FILE_OPEN_CREAT;
	if ((flags & O_TRUNC ) == O_TRUNC ) fileio_flags |= IO_FILE_OPEN_TRUNC;
	if ((flags & O_EXCL  ) == O_EXCL  ) fileio_flags |= IO_FILE_OPEN_EXCL;
	comm_buf[0] = (char)IO_FILE_OPEN_REQUEST;
	comm_buf[1] = fileio_flags;
	__comm_io_send_to_host(comm_buf, 2);
	size_t_to_binary(len, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(len));
	__comm_io_send_to_host(name, len);
	__comm_io_receive_from_host(comm_buf, 1);
	if (comm_buf[0] == IO_COMMON_ACK) {
		__comm_io_receive_from_host(comm_buf, sizeof(int));
		return binary_to_int(comm_buf);
	} else {
		return -1;
	}
}

int __multi_io_file_close(int fd)
{
	char comm_buf[sizeof(int)];
	comm_buf[0] = (char)IO_FILE_CLOSE_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	int_to_binary(fd, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(fd));
	__comm_io_receive_from_host(comm_buf, 1);
	return (comm_buf[0] == IO_COMMON_ACK) ? 0 : -1;
}

int __multi_io_file_read(int fd, void* ret_buf, size_t size)
{
	char comm_buf[sizeof(int)];
	comm_buf[0] = (char)IO_FILE_READ_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	int_to_binary(fd, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(fd));
	__comm_io_receive_from_host(comm_buf, 1);
	if (comm_buf[0] == IO_COMMON_ACK) {
		char* p = (char*)ret_buf;
		size_t_to_binary(size, comm_buf);
		__comm_io_send_to_host(comm_buf, sizeof(size));
		while (1) {
			size_t t;
			__comm_io_receive_from_host(comm_buf, sizeof(t));
			t = binary_to_size_t(comm_buf);
			if (t == 0) break;
			__comm_io_receive_from_host(p, t);
			p += t;
		}
		return (int)(p - (char*)ret_buf);
	} else {
		return -1;
	}
}

int __multi_io_file_write(int fd, const void* value, size_t size)
{
	char comm_buf[sizeof(int)];
	comm_buf[0] = (char)IO_FILE_WRITE_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	int_to_binary(fd, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(fd));
	__comm_io_receive_from_host(comm_buf, 1);
	if (comm_buf[0] == IO_COMMON_ACK) {
		size_t_to_binary(size, comm_buf);
		__comm_io_send_to_host(comm_buf, sizeof(size));
		__comm_io_send_to_host(value, size);
		__comm_io_receive_from_host(comm_buf, sizeof(int));
		return binary_to_int(comm_buf);
	} else {
		return -1;
	}
}

off_t __multi_io_file_lseek(int fd, off_t offset, int whence)
{
	char comm_buf[sizeof(int)];
	comm_buf[0] = (char)IO_FILE_LSEEK_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	int_to_binary(fd, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(fd));
	__comm_io_receive_from_host(comm_buf, 1);
	if (comm_buf[0] == IO_COMMON_ACK) {
		int_to_binary((int)offset, comm_buf);
		__comm_io_send_to_host(comm_buf, sizeof(offset));
		switch (whence) {
		case SEEK_SET:
			comm_buf[0] = (char)IO_FILE_SEEK_SET;
			break;
		case SEEK_CUR:
			comm_buf[0] = (char)IO_FILE_SEEK_CUR;
			break;
		case SEEK_END:
			comm_buf[0] = (char)IO_FILE_SEEK_END;
			break;
		default:
			/* input error */
			comm_buf[0] = (char)IO_FILE_SEEK_SET;
			break;
		}
		__comm_io_send_to_host(comm_buf, 1);
		__comm_io_receive_from_host(comm_buf, sizeof(int));
		return (off_t)binary_to_int(comm_buf);
	} else {
		return -1;
	}
}

int __multi_io_commandline_read_argc(void)
{
	char comm_buf[sizeof(int)];
	comm_buf[0] = (char)IO_COMMANDLINE_READ_ARGC_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	__comm_io_receive_from_host(comm_buf, 1);
	if (comm_buf[0] == (char)IO_COMMON_ACK) {
		__comm_io_receive_from_host(comm_buf, sizeof(int));
		return binary_to_int(comm_buf);
	} else {
		return -1;
	}
}

void __multi_io_commandline_read_argv(int index, size_t ret_max,
	size_t* ret_len, char* ret)
{
	char comm_buf[sizeof(int)];
	comm_buf[0] = (char)IO_COMMANDLINE_READ_ARGV_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	int_to_binary(index, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(index));
	__comm_io_receive_from_host(comm_buf, 1);
	if (comm_buf[0] == (char)IO_COMMON_ACK) {
		size_t len;
		__comm_io_receive_from_host(comm_buf, sizeof(len));
		len = binary_to_size_t(comm_buf);
		if (len < ret_max - 1) {
			__comm_io_receive_from_host(ret, len);
			ret[len] = '\0';
			*ret_len = len;
		} else {
			while (len > 0) {
				size_t t = (len < ret_max) ? len : ret_max;
				__comm_io_receive_from_host(ret, t);
				len -= t;
			}
			ret[0] = '\0';
			*ret_len = 0;
		}
	} else {
		ret[0] = '\0';
		*ret_len = 0;
	}
}

int __multi_io_commandline_write_status(int value)
{
	char comm_buf[sizeof(int)];
	comm_buf[0] = (char)IO_COMMANDLINE_WRITE_STATUS_REQUEST;
	__comm_io_send_to_host(comm_buf, 1);
	int_to_binary(value, comm_buf);
	__comm_io_send_to_host(comm_buf, sizeof(value));
	__comm_io_receive_from_host(comm_buf, 1);
	return (comm_buf[0] == (char)IO_COMMON_ACK) ? 1 : 0;
}
