/*
 * <<< mpni_io_funcs.c >>>
 *
 * --- Communication routines with isis::mp_network_interface
 *     'mpni_io_funcs'
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include "multi_io_funcs.h"
#include "mpni_io_funcs.h"
#include "io_protocol.h"

#define MPNI_IO_FUNCS_DEBUG

static int __mpni_init_flag = 0;
static volatile unsigned int* __mpni_address;
static size_t __mpni_max_data_size;
static size_t __mpni_send_request_buffer_size;
static size_t __mpni_receive_request_buffer_size;
static size_t __mpni_receive_pool_size;
static size_t __mpni_sendflags_offset;
static size_t __mpni_recvflags_offset;

int __mpni_io_init(void)
{
	int t;
	if (__mpni_init_flag) return 1;
	if (!__multi_io_sysinfo_read_integer("network_interface_address", &t)) {
		return 0; // error: cannot address NI
	}
	__mpni_address = (volatile unsigned int*)(t);
	__mpni_max_data_size
		= __mpni_address[MPNI_IO_OFFSET_MAX_DATA_SIZE];
	__mpni_send_request_buffer_size
		= __mpni_address[MPNI_IO_OFFSET_SENDREQBUF_SIZE];
	__mpni_receive_request_buffer_size
		= __mpni_address[MPNI_IO_OFFSET_RECVREQBUF_SIZE];
	__mpni_receive_pool_size
		= __mpni_address[MPNI_IO_OFFSET_RECVPOOL_SIZE];
	__mpni_sendflags_offset
		= __mpni_address[MPNI_IO_OFFSET_SENDFLAGS_OFFSET];
	__mpni_recvflags_offset
		= __mpni_address[MPNI_IO_OFFSET_RECVFLAGS_OFFSET];
	__mpni_init_flag = 1;
	return 1;
}

void __mpni_io_finalize(void)
{
}

void __mpni_io_sense(void)
{
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_SENSE;
}

int __mpni_io_send(size_t dst, void* header, size_t hsz, void* data,
				   size_t dsz, size_t* ret_id)
{
	size_t id;
	__mpni_address[MPNI_IO_OFFSET_PARTNER_ADDRESS] = (unsigned int)(dst);
	__mpni_address[MPNI_IO_OFFSET_HEADER_ADDRESS] = (unsigned int)(header);
	__mpni_address[MPNI_IO_OFFSET_HEADER_SIZE] = (unsigned int)(hsz);
	__mpni_address[MPNI_IO_OFFSET_DATA_ADDRESS] = (unsigned int)(data);
	__mpni_address[MPNI_IO_OFFSET_DATA_SIZE] = (unsigned int)(dsz);
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_SEND;
	id = __mpni_address[MPNI_IO_OFFSET_REQUEST_ID];
	if (id < __mpni_send_request_buffer_size) {
		*ret_id = id;
		return 1; /* success */
	} else {
		return 0; /* failed */
	}
	/* not reached */
}

int __mpni_io_rawsend(size_t dst, void* data, size_t dsz, size_t* ret_id)
{
	size_t id;
	__mpni_address[MPNI_IO_OFFSET_PARTNER_ADDRESS] = (unsigned int)(dst);
	__mpni_address[MPNI_IO_OFFSET_DATA_ADDRESS] = (unsigned int)(data);
	__mpni_address[MPNI_IO_OFFSET_DATA_SIZE] = (unsigned int)(dsz);
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_RAWSEND;
	id = __mpni_address[MPNI_IO_OFFSET_REQUEST_ID];
	if (id < __mpni_send_request_buffer_size) {
		*ret_id = id;
		return 1; /* success */
	} else {
		return 0; /* failed */
	}
	/* not reached */
}

int __mpni_io_psend(size_t dst, void* header, size_t hsz, void* data,
					size_t dsz, size_t* ret_id)
{
	size_t id;
	__mpni_address[MPNI_IO_OFFSET_PARTNER_ADDRESS] = (unsigned int)(dst);
	__mpni_address[MPNI_IO_OFFSET_HEADER_ADDRESS] = (unsigned int)(header);
	__mpni_address[MPNI_IO_OFFSET_HEADER_SIZE] = (unsigned int)(hsz);
	__mpni_address[MPNI_IO_OFFSET_DATA_ADDRESS] = (unsigned int)(data);
	__mpni_address[MPNI_IO_OFFSET_DATA_SIZE] = (unsigned int)(dsz);
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_PSEND;
	id = __mpni_address[MPNI_IO_OFFSET_REQUEST_ID];
	if (id < __mpni_send_request_buffer_size) {
		*ret_id = id;
		return 1; /* success */
	} else {
		return 0; /* failed */
	}
	/* not reached */
}

int __mpni_io_prawsend(size_t dst, void* data, size_t dsz, size_t* ret_id)
{
	size_t id;
	__mpni_address[MPNI_IO_OFFSET_PARTNER_ADDRESS] = (unsigned int)(dst);
	__mpni_address[MPNI_IO_OFFSET_DATA_ADDRESS] = (unsigned int)(data);
	__mpni_address[MPNI_IO_OFFSET_DATA_SIZE] = (unsigned int)(dsz);
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_PRAWSEND;
	id = __mpni_address[MPNI_IO_OFFSET_REQUEST_ID];
	if (id < __mpni_send_request_buffer_size) {
		*ret_id = id;
		return 1; /* success */
	} else {
		return 0; /* failed */
	}
	/* not reached */
}

int __mpni_io_recv(size_t src, void* header, size_t hsz, void* data,
				   size_t dsz, int anysrc_flag, size_t* ret_id)
{
	int latebind_flag = (data == NULL);
	size_t id;
	unsigned int flags;
	flags = (anysrc_flag ? MPNI_IO_FLAGS_MASK_ANYSRC : 0)
		  | (latebind_flag ? MPNI_IO_FLAGS_MASK_LATEBIND : 0);
	if (!anysrc_flag) {
		__mpni_address[MPNI_IO_OFFSET_PARTNER_ADDRESS] = (unsigned int)(src);
	}
	__mpni_address[MPNI_IO_OFFSET_HEADER_ADDRESS] = (unsigned int)(header);
	__mpni_address[MPNI_IO_OFFSET_HEADER_SIZE] = (unsigned int)(hsz);
	if (!latebind_flag) {
		__mpni_address[MPNI_IO_OFFSET_DATA_ADDRESS] = (unsigned int)(data);
	}
	__mpni_address[MPNI_IO_OFFSET_DATA_SIZE] = (unsigned int)(dsz);
	__mpni_address[MPNI_IO_OFFSET_FLAGS] = flags;
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_RECV;
	id = __mpni_address[MPNI_IO_OFFSET_REQUEST_ID];
	if (id < __mpni_receive_request_buffer_size) {
		*ret_id = id;
		return 1; /* success */
	} else {
		return 0; /* failed */
	}
	/* not reached */
}

int __mpni_io_rawrecv(size_t src, void* data, size_t dsz, int anysrc_flag,
					  int anysize_flag, size_t* ret_id)
{
	int cancel_flag = (data == NULL);
	size_t id;
	unsigned int flags;
	flags = (anysrc_flag ? MPNI_IO_FLAGS_MASK_ANYSRC : 0)
		  | (anysize_flag ? MPNI_IO_FLAGS_MASK_ANYSIZE : 0)
		  | (cancel_flag ? MPNI_IO_FLAGS_MASK_CANCEL : 0);
	if (!anysrc_flag) {
		__mpni_address[MPNI_IO_OFFSET_PARTNER_ADDRESS] = (unsigned int)(src);
	}
	if (!cancel_flag) {
		__mpni_address[MPNI_IO_OFFSET_DATA_ADDRESS] = (unsigned int)(data);
	}
	__mpni_address[MPNI_IO_OFFSET_DATA_SIZE] = (unsigned int)(dsz);
	__mpni_address[MPNI_IO_OFFSET_FLAGS] = flags;
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_RAWRECV;
	id = __mpni_address[MPNI_IO_OFFSET_REQUEST_ID];
	if (id < __mpni_receive_request_buffer_size) {
		*ret_id = id;
		return 1; /* success */
	} else {
		return 0; /* failed */
	}
	/* not reached */
}

int __mpni_io_precv(size_t src, void* header, size_t hsz, void* data,
					size_t dsz, int anysrc_flag, size_t* ret_id)
{
	int latebind_flag = (data == NULL);
	size_t id;
	unsigned int flags;
	flags = (anysrc_flag ? MPNI_IO_FLAGS_MASK_ANYSRC : 0)
		  | (latebind_flag ? MPNI_IO_FLAGS_MASK_LATEBIND : 0);
	if (!anysrc_flag) {
		__mpni_address[MPNI_IO_OFFSET_PARTNER_ADDRESS] = (unsigned int)(src);
	}
	__mpni_address[MPNI_IO_OFFSET_HEADER_ADDRESS] = (unsigned int)(header);
	__mpni_address[MPNI_IO_OFFSET_HEADER_SIZE] = (unsigned int)(hsz);
	if (!latebind_flag) {
		__mpni_address[MPNI_IO_OFFSET_DATA_ADDRESS] = (unsigned int)(data);
	}
	__mpni_address[MPNI_IO_OFFSET_DATA_SIZE] = (unsigned int)(dsz);
	__mpni_address[MPNI_IO_OFFSET_FLAGS] = flags;
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_PRECV;
	id = __mpni_address[MPNI_IO_OFFSET_REQUEST_ID];
	if (id < __mpni_receive_request_buffer_size) {
		*ret_id = id;
		return 1; /* success */
	} else {
		return 0; /* failed */
	}
	/* not reached */
}

int __mpni_io_prawrecv(size_t src, void* data, size_t dsz, int anysrc_flag,
					   int anysize_flag, size_t* ret_id)
{
	int cancel_flag = (data == NULL);
	size_t id;
	unsigned int flags;
	flags = (anysrc_flag ? MPNI_IO_FLAGS_MASK_ANYSRC : 0)
		  | (anysize_flag ? MPNI_IO_FLAGS_MASK_ANYSIZE : 0)
		  | (cancel_flag ? MPNI_IO_FLAGS_MASK_CANCEL : 0);
	if (!anysrc_flag) {
		__mpni_address[MPNI_IO_OFFSET_PARTNER_ADDRESS] = (unsigned int)(src);
	}
	if (!cancel_flag) {
		__mpni_address[MPNI_IO_OFFSET_DATA_ADDRESS] = (unsigned int)(data);
	}
	__mpni_address[MPNI_IO_OFFSET_DATA_SIZE] = (unsigned int)(dsz);
	__mpni_address[MPNI_IO_OFFSET_FLAGS] = flags;
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_PRAWRECV;
	id = __mpni_address[MPNI_IO_OFFSET_REQUEST_ID];
	if (id < __mpni_receive_request_buffer_size) {
		*ret_id = id;
		return 1; /* success */
	} else {
		return 0; /* failed */
	}
	/* not reached */
}

int __mpni_io_latebind(size_t id, void* data)
{
	size_t ret_id;
	__mpni_address[MPNI_IO_OFFSET_REQUEST_ID] = (unsigned int)id;
	__mpni_address[MPNI_IO_OFFSET_DATA_ADDRESS] = (unsigned int)(data);
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_LATEBIND;
	ret_id = (unsigned int)(__mpni_address[MPNI_IO_OFFSET_REQUEST_ID]);
	return (ret_id == id);
}

int __mpni_io_latecancel(size_t id)
{
	size_t ret_id;
	__mpni_address[MPNI_IO_OFFSET_REQUEST_ID] = (unsigned int)id;
	__mpni_address[MPNI_IO_OFFSET_REQUEST_TYPE] = MPNI_IO_LATECANCEL;
	ret_id = (unsigned int)(__mpni_address[MPNI_IO_OFFSET_REQUEST_ID]);
	return (ret_id == id);
}

int __mpni_io_send_poll(size_t id)
{
	return __mpni_address[__mpni_sendflags_offset + id];
}

int __mpni_io_recv_poll(size_t id)
{
	return __mpni_address[__mpni_recvflags_offset + id];
}

#ifdef MPNI_IO_FUNCS_DEBUG

#include <stdio.h>

void __mpni_io_show_fixed_parameters(void)
{
	printf("mpni address:     %p\n", __mpni_address);
	printf("max data size:    %d\n", __mpni_max_data_size);
	printf("sendreqbuf size:  %d\n", __mpni_send_request_buffer_size);
	printf("recvreqbuf size:  %d\n", __mpni_receive_request_buffer_size);
	printf("recvpool size:    %d\n", __mpni_receive_pool_size);
	printf("sendflags offset: %d\n", __mpni_sendflags_offset);
	printf("recvflags offset: %d\n", __mpni_recvflags_offset);
}

void __mpni_io_show_variable_parameters(void)
{
	size_t sendreq_count, recvreq_count, recvpool_count;
	__mpni_io_sense();
	sendreq_count = __mpni_address[MPNI_IO_OFFSET_PENDING_SENDREQ_COUNT];
	recvreq_count = __mpni_address[MPNI_IO_OFFSET_PENDING_RECVREQ_COUNT];
	recvpool_count = __mpni_address[MPNI_IO_OFFSET_PENDING_RECVPOOL_COUNT];
	printf("sendreq count:  %d\n", sendreq_count);
	printf("recvreq count:  %d\n", recvreq_count);
	printf("recvpool count: %d\n", recvpool_count);
}

#endif /* MPNI_IO_FUNCS_DEBUG */
