`include "def.h"
module rv32i(
input clk, rst_n,
input [`DATA_W-1:0] instr,
input [`DATA_W-1:0] readdata,
output reg [`DATA_W-1:0] pc, 
output [`DATA_W-1:0] adrdata,
output [`DATA_W-1:0] writedata,
output we,
output ecall);

/*  Instruction Fetch Stage */
reg [`DATA_W-1:0] instrD ;

always @(posedge clk or negedge rst_n)
begin
   if(!rst_n) instrD <= 0;
   else instrD <= instr;
end

always @(posedge clk or negedge rst_n)
begin
   if(!rst_n) pc <= 0;
   else
     pc <= pc+4;
end

/*  Instruction Decorder Stage */

wire addcom;
wire [2:0] funct3;
wire [6:0] funct7;
wire [`REG_W-1:0] rs1, rs2, rd;
wire [`DATA_W-1:0] reg1, reg2;
wire [`OPCODE_W-1:0] opcode;
wire [`SHAMT_W-1:0] shamt;
wire [`OPCODE_W-1:0] func;
wire [`DATA_W-1:0] pcplus4;
wire [11:0] imm_i, imm_s;
wire [12:0] imm_b;
wire [20:0] imm_j, imm_u;
wire rwe;
wire alu_op, imm_op;
wire sw_op;
wire slt_op, ecall_op;
wire lui_op;
wire ext;
wire signed [31:0] sreg1, sreg2;
wire [19:0] sext;
wire [`DATA_W-1:0] imm;
// pipeline registers
reg [`DATA_W-1:0] immE, reg1E, reg2E;
wire [`DATA_W-1:0] resultdata;
reg [`REG_W-1:0] rdE,rdW ;
reg [2:0] funct3E;
reg extE, sw_opE, lw_opE, lui_opE, ecall_opE, rweE, addcomE;
reg rweW, alu_opE;

assign sreg1 = $signed(reg1);
assign sreg2 = $signed(reg2);
assign {funct7, rs2, rs1, funct3, rd, opcode} = instrD;
assign sext = {20{instrD[31]}};
assign imm_i = {funct7,rs2};
assign imm_s = {funct7,rd};
assign imm_b = {funct7[6],rd[0],funct7[5:0],rd[4:1],1'b0};
assign imm_j = {instrD[31], instrD[19:12],instrD[20],instrD[30:21],1'b0};
assign imm_u = instrD[31:12];
// Decorder
assign sw_op = (opcode == `OP_STORE) & (funct3 == 3'b010);
assign lw_op = (opcode == `OP_LOAD) & (funct3 == 3'b010);
assign alu_op = (opcode == `OP_REG) ;
assign imm_op = (opcode == `OP_IMM) ;
assign lui_op = (opcode == `OP_LUI);
assign ecall_op = (opcode == `OP_SPE) & (funct3 == 3'b000);
assign ext = alu_op  & funct7[5];

assign	imm = imm_op | lw_op ? {sext, imm_i}: 
				sw_op ? {sext, imm_s}: 
				lui_op ? {imm_u,12'b0}: {sext[10:0], imm_j}; 
				
assign rwe = lw_op | alu_op | imm_op | lui_op ;
assign addcom = (lw_op|sw_op);

rfile rfile_1(.clk(clk), .rd1(reg1), .a1(rs1), .rd2(reg2), .a2(rs2), 
	.wd3(resultdata), .a3(rdW), .we3(rweW));

always @(posedge clk or negedge rst_n) begin
	if(!rst_n) begin
		reg1E <=0; reg2E <=0;
		rdE <=0; funct3E <=0;
		sw_opE <= 0; lw_opE <= 0; 
		ecall_opE <= 0; lui_opE <= 0;
		alu_opE<=0; 
		rweE <= 0; 
		addcomE <= 0; extE <= 0;
		immE <=0; end
	else  begin
		reg1E <=reg1; reg2E <=reg2;
		rdE <=rd; funct3E <=funct3;
		sw_opE <= sw_op; lw_opE <= lw_op; 
		ecall_opE <= ecall_op; lui_opE <= lui_op;
		alu_opE<=alu_op; 
		rweE <= rwe; 
		addcomE <= addcom; extE <= ext;
		immE <=imm; end
end

/*  Execution Stage */
wire [`DATA_W-1:0] srca, srcb, result, aluresult;
reg [`DATA_W-1:0] resultM, reg2M;
reg sw_opM, lw_opM, ecall_opM, rweM;
reg [`REG_W-1:0] rdM ;
assign srca = reg1E;
assign srcb = alu_opE? reg2E: immE;

assign result = lui_opE ? immE: aluresult ;

alu alu_1(.a(srca), .b(srcb), .s(funct3E), .ext(extE), .addcom(addcomE),  .y(aluresult));

always @(posedge clk or negedge rst_n) begin
	if(!rst_n) begin
		resultM <= 0; reg2M <= 0;
		rdM <=0; rweM <=0; sw_opM <= 0; 
		lw_opM <=0; ecall_opM <= 0; end
	else begin
		resultM <= result; reg2M <= reg2E;
		rdM <=rdE; rweM <=rweE; sw_opM <= sw_opE; 
		lw_opM <=lw_opE; ecall_opM <= ecall_opE; end
end
		
/*  Memory Access Stage */
reg lw_opW, ecall_opW;
reg [`DATA_W-1:0] readdataW;
reg [`DATA_W-1:0] resultW;
assign we = sw_opM;
assign adrdata = resultM;
assign writedata = reg2M;

always @(posedge clk or negedge rst_n) begin
	if(!rst_n) begin
		lw_opW <=0; ecall_opW<= 0;
		rdW <= 0; rweW <= 0;
		resultW <= 0;
		readdataW<=0; end
	else  begin
		lw_opW <=lw_opM; ecall_opW<= ecall_opM;
		rdW <= rdM; rweW <= rweM;
		readdataW <= readdata;
		resultW <= resultM; end
end
/*  Write back Stage */
assign ecall = ecall_opW;	
assign resultdata = lw_opW ? readdataW : resultW;
endmodule
